/*
| IBIC Simulation Tool
| Copyright (C) 2012-2013 Jacopo Forneris
| http://www.dfs.unito.it/solid/IST.html

This file is part of IBIC simulation tool.

    IBIC simulation tool is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    IBIC simulation tool is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with IBIC Simulation Tool.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "mainwindow.h"
#include "filesdialog.h"
#include "ui_filesdialog.h"

FilesDialog::FilesDialog(QWidget *parent) :
    QDialog(parent),
    ui(new Ui::FilesDialog)
{


    ui->setupUi(this);
    MainWindow *obj = (MainWindow *) qApp->activeWindow();


    //check whether dialog has been used before

    if(obj->checkdialog==1)
    {

        //intialize displays values previously stored in the program
        //converts everything from [m] to the proper unit of measure
        temp=obj->temp;
        mun=obj->mun*1e4;
        mup=obj->mup*1e4;
        vsatp=obj->vsatp*100;
        vsatn=obj->vsatn*100;
        taup=obj->taup;
        taun=obj->taun;
        path_gunn=obj->path_gunn;
        path_efield=obj->path_efield;

        ui->lineEdit_Efield->setText(path_efield);
        ui->lineEdit_Gunn->setText(path_gunn);
        ui->lineEdit_mun->setText(QString::number(mun));
        ui->lineEdit_mup->setText(QString::number(mup));
        ui->lineEdit_taun->setText(QString::number(taun));
        ui->lineEdit_taup->setText(QString::number(taup));
        ui->lineEdit_temp->setText(QString::number(temp));
        ui->lineEdit_vsatn->setText(QString::number(vsatn));
        ui->lineEdit_vsatp->setText(QString::number(vsatp));

        //displays electric field and weightinh potential preview
        plotEfield();
        plotGunn();

        ui->pushButton_OK->setEnabled(true);
    }

        //if this is the first time, unckeck control variable and set device length to zero
    else
    {
        checkdialog=0;
        ui->pushButton_OK->setEnabled(false);
        L=0;
    }

    //any modification of text boxes in the window launches the update_values() for all variables
    connect(ui->lineEdit_mup,SIGNAL(editingFinished()),this,SLOT(update_values()));
    connect(ui->lineEdit_mun,SIGNAL(editingFinished()),this,SLOT(update_values()));
    connect(ui->lineEdit_taup,SIGNAL(editingFinished()),this,SLOT(update_values()));
    connect(ui->lineEdit_taun,SIGNAL(editingFinished()),this,SLOT(update_values()));
    connect(ui->lineEdit_temp,SIGNAL(editingFinished()),this,SLOT(update_values()));
    connect(ui->lineEdit_vsatp,SIGNAL(editingFinished()),this,SLOT(update_values()));
    connect(ui->lineEdit_vsatn,SIGNAL(editingFinished()),this,SLOT(update_values()));

}

FilesDialog::~FilesDialog()
{
    delete ui;
}

void FilesDialog::on_pushButton_efield_clicked()
{
    //open browser window
    FileBrowser browser;
    browser.exec();

    //store path to electric field input file
    path_efield=browser.on_buttonBox_accepted();
    ui->lineEdit_Efield->setText(path_efield);

    //plot the profile and print in the dialog length and space step
    plotEfield();
    text_efield();
}



    void FilesDialog::plotEfield()
    {
    //define text stream for "read from file" operations
    QFile file(path_efield);
    QTextStream its(&file);
    QString line;

    efilenotfound=0;

    if (!file.open(QIODevice::ReadOnly))
    {
        efilenotfound=1;
    }

    //set to zero size for pushback coordinate and electric field vector
    efield.resize(0);
    pos_efield.resize(0);
    int i=0;
    space_step_efield=0;
    int array_step_count=0;

    //define aux string for input data processing
    QStringList list;

    while( !its.atEnd() )
    {
        //import line, check punctuation, split columns
        line = its.readLine();
        line.replace(",",".");
        line.replace("\t", "    ");
        list=line.split(" ",QString::SkipEmptyParts);

        //check line format to skip header or corrupted lines
        if(list.size()>0)
        {
            bool validate1, validate2;
            list.first().toDouble(&validate1);
            list.last().toDouble(&validate2);

            //check for numeric input data and push back to vectors
            if (validate1 == true && validate2 ==true && list.size()>1)
            {
                pos_efield.push_back(list.first().toDouble()); // units: [m]
                efield.push_back(list.last().toDouble());  //units [V/cm]

                //check that mesh step size is constant. if so, store its value. if not, increment a control variable
                if(i==1)
                {
                    space_step_efield=(pos_efield[i]-pos_efield[i-1]);
                }
                if(i>0)
                {
                    if(fabs(pos_efield[i]-pos_efield[i-1] -space_step_efield) >1e-15 ) {array_step_count++;}
                }
                i++;
            }
        }


    }

    file.close();

    //overall control variable for constant mesh size (for dialog acceptance)
    int control_deltax_efield=-1;
    if (array_step_count==0)
      {
        //store mesh step size
        deltax=space_step_efield;
        control_deltax_efield=0;
      }
    //store device length
    L=efield.size();

    //plot electric field profile and update dialog
    setupEfield(ui->efield_plot);
    ui->efield_plot->replot();
    update_values();
}


void FilesDialog::on_pushButton_gunn_clicked()
{
  //open browser window
  FileBrowser browser;
  browser.exec();

  //store path to weighting potential input file
  path_gunn=browser.on_buttonBox_accepted();
  ui->lineEdit_Gunn->setText(path_gunn);

  //plot the profile and print in the dialog length and space step
  plotGunn();
  text_gunn();
}

 void FilesDialog::plotGunn()
{

  //define text stream for "read from file" operations
  QFile file_gunn(path_gunn);
  QTextStream its_gunn(&file_gunn);
  QString line_gunn;

  gfilenotfound=0;
  if (!file_gunn.open(QIODevice::ReadOnly))
  {
        gfilenotfound=1;
  }

  //set to zero size for pushback coordinate and electric field vector
  gunn.resize(0);
  pos_gunn.resize(0);
  int i=0;
  space_step_gunn=0;
  int array_step_count=0;

  //define aux string for input data processing
  QStringList list_gunn;

  while( !its_gunn.atEnd() )
  {
      //import line, check punctuation, split columns
      line_gunn = its_gunn.readLine();
      line_gunn.replace(",",".");
      line_gunn.replace("\t", "    ");
      list_gunn=line_gunn.split(" ",QString::SkipEmptyParts);

      //check line format to skip header or corrupted lines
      if(list_gunn.size()>0)
      {
          bool validate1, validate2;
          list_gunn.first().toDouble(&validate1);
          list_gunn.last().toDouble(&validate2);

          //check for numeric input data and push back to vectors
          if (validate1 == true && validate2 ==true && list_gunn.size()>1)
          {
              pos_gunn.push_back(list_gunn[0].toDouble()); //units: [m]
              gunn.push_back(list_gunn[1].toDouble()); //units: [ ]

              //check that mesh step size is constant. if so, store its value. if not, increment a control variable
              if(i==1)
              {
                  space_step_gunn=(pos_gunn[i]-pos_gunn[i-1]);
              }
              if(i>0)
              {
                    if(fabs(pos_gunn[i]-pos_gunn[i-1] -space_step_gunn) >1e-15 ){array_step_count++;}
              }
              i++;
          }
      }
  }

  file_gunn.close();

  //overall control variable for constant mesh size (for dialog acceptance)
  int control_deltax_gunn=-1;
  if (array_step_count==0)
    {
      //store mesh step size
      deltax=space_step_gunn;
      control_deltax_gunn=0;
    }

  //store device length (if not done by electric field first)
  if(L==0)
  {
      L=gunn.size();
  }

  //plot electric field profile and update dialog
  setupGunn(ui->gunn_plot);
  ui->gunn_plot->replot();
  update_values();
}


void FilesDialog::setupEfield(QCustomPlot *potPlot)
{
    //add graph, set blue line and green filling brush color
    potPlot->addGraph();
    potPlot->graph(0)->setBrush(QBrush(QColor(0, 250, 0)));
    potPlot->graph(0)->setAntialiasedFill(false);
    potPlot->graph(0)->setPen(QPen(Qt::blue)); // line color blue for first graph

    //convert coordinate from [m] to [um], and find x- and y- max values
    QVector<double> x(efield.size());
    double emax=0;
    double emin=0;
    for (int i=0; i<efield.size(); ++i)
    {
        x[i] = i*deltax*1e6;
        if(efield[i]>emax){emax=efield[i];}
        if(efield[i]<emin){emin=efield[i];}
    }

    // configure right and top axis
    potPlot->xAxis->setVisible(true);
    potPlot->xAxis2->setVisible(false);
    potPlot->xAxis2->setTickLabels(true);
    potPlot->yAxis2->setVisible(true);
    potPlot->yAxis2->setTickLabels(false);
    potPlot->graph(0)->setData(x, efield);
    potPlot->xAxis->setRange(-0.0,L*deltax*1e6);
    potPlot->yAxis->setRange(emin*1.2,emax*1.2);
    potPlot->xAxis->setLabel("Position (um)");
    potPlot->yAxis->setLabel("Electric field (V/cm)");
    potPlot->yAxis->setAutoTickCount(3);
}



void FilesDialog::setupGunn(QCustomPlot *potPlot)
{
        //add graph, set line and filling brush color
        potPlot->addGraph();
        potPlot->graph(0)->setBrush(QBrush(QColor(0, 250, 0)));
        potPlot->graph(0)->setAntialiasedFill(false);
        potPlot->graph(0)->setPen(QPen(Qt::blue));

        //convert coordinate from [m] to [um], and find x- and y- max values
        QVector<double> x(gunn.size());
        double gunnmax=0;
        double gunnmin=0;
        for (int i=0; i<gunn.size(); ++i)
        {
            x[i] = i*deltax*1e6;
            if(gunn[i]>gunnmax){gunnmax=gunn[i];}
            if(gunn[i]<gunnmin){gunnmin=gunn[i];}
        }

      // configure right and top axis      potPlot->xAxis->setVisible(true);
      potPlot->xAxis2->setVisible(false);
      potPlot->xAxis2->setTickLabels(true);
      potPlot->yAxis2->setVisible(true);
      potPlot->yAxis2->setTickLabels(false);
      potPlot->graph(0)->setData(x, gunn);
      potPlot->graph(0)->rescaleAxes(true);
      potPlot->xAxis->setLabel("Position (um)");
      potPlot->yAxis->setLabel("Weighting potential");
      potPlot->yAxis->setRange(gunnmin,gunnmax);
      potPlot->xAxis->setRange(-0.0,L*deltax*1e6);
      potPlot->xAxis->setAutoTickCount(2);
}


void FilesDialog::on_pushButton_OK_clicked()
{
        //store dialog text input to be used from main program
        path_efield = ui ->lineEdit_Efield->text();
        path_gunn = ui->lineEdit_Gunn->text();
        temp = ui->lineEdit_temp->text().toDouble();
        mun = ui->lineEdit_mun-> text().toDouble();
        mup = ui->lineEdit_mup-> text().toDouble();
        vsatp = ui->lineEdit_vsatp-> text().toDouble();
        vsatn = ui->lineEdit_vsatn-> text().toDouble();
        taup = ui->lineEdit_taup-> text().toDouble();
        taun = ui->lineEdit_taun-> text().toDouble();

        //check control variable indicating that the dialog has been used at least once
        checkdialog=1;

}


void FilesDialog::on_pushButton_cancel_clicked()
{

}


void FilesDialog::update_values()
{
    //store updated value for text input variables
    path_efield = ui ->lineEdit_Efield->text();
    path_gunn = ui->lineEdit_Gunn->text();
    temp = ui->lineEdit_temp->text().toDouble();
    mun = ui->lineEdit_mun-> text().toDouble();
    mup = ui->lineEdit_mup-> text().toDouble();
    vsatp = ui->lineEdit_vsatp-> text().toDouble();
    vsatn = ui->lineEdit_vsatn-> text().toDouble();
    taup = ui->lineEdit_taup-> text().toDouble();
    taun = ui->lineEdit_taun-> text().toDouble();

    //check that weighting potential and electric field are compatible
    //check that all input have been provided
    //if so, enable dialog OK button
    if(gunn.size()==efield.size() && L>0 && mup>0 && mun>0 && vsatp>0 && vsatn>0 && taup>0 && taun>0 && temp >=0 )    //DA SPOSTARE SUL BOTTONE OK, o come controllo da qualche parte
    {
         ui->pushButton_OK->setEnabled(true);
    }
}


void FilesDialog::text_efield(){

    //print in dialog window relevant info on ascii import
    //length and space step
    ui->textBrowser_efield->clear();
    if(efilenotfound==0)
    {
        ui->textBrowser_efield->setText("length:" + QString::number(efield.size()) + " steps" + "\t" + "deltax: " + QString::number(space_step_efield) + " m");
    }
    if(efilenotfound==1)
    {
        ui->textBrowser_efield->setText("File not found");
    }
}


void FilesDialog::text_gunn()
{
    ui->textBrowser_gunn->clear();
    if(gfilenotfound==0)
    {
        ui->textBrowser_gunn->setText("length:" + QString::number(gunn.size()) + " steps" + "\t" + "deltax: " + QString::number(space_step_gunn) + " m");
    }
    if(gfilenotfound==1)
    {
        ui->textBrowser_gunn->setText("File not found");
    }
}
