/*
| IBIC Simulation Tool
| Copyright (C) 2012-2013 Jacopo Forneris
| http://www.dfs.unito.it/solid/IST.html

This file is part of IBIC simulation tool.

    IBIC simulation tool is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    IBIC simulation tool is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with IBIC Simulation Tool.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "mainwindow.h"
#include "ui_mainwindow.h"
#include <QDebug>
#include <math.h>
#include <fstream>
#include <stdio.h>
#include <sys/time.h>
#include <iomanip>
#include <string>
#include <sstream>
#include <QString.h>
#include <string.h>
#include <iostream>
#include <stdlib.h>
#include <qfile.h>
#include <qtextstream.h>
#include <QTextBrowser>
#include "generators.h"
#include <QDir>
#include <QThread>
#include <QtConcurrent/QtConcurrentRun>
#include <MyLibraryWrapper.h>

using namespace std;



MyLibraryWrapper *wrapper = new MyLibraryWrapper();

MyLibraryWrapper::MyLibraryWrapper() : QThread()
{
  // We have to do this to make sure our thread has the
  // correct affinity.
  moveToThread(this);


  // This will do nothing until the user calls start().
}

void MyLibraryWrapper::run()
{
  // This schedules the doTheWork() function
  // to run just after our event loop starts up
     //   MainWindow *w;
  QTimer::singleShot(0, this, SLOT(doTheWork()));

  // This starts the event loop. Note that
  // exec() does not return until the
  // event loop is stopped.
  exec();

}



void MyLibraryWrapper::launch()
{
    run();
}

void MyLibraryWrapper::doTheWork()
{
}

void MainWindow::wrapperDone()
{
wrapper->dataTimer.stop();
ui->pushButton_stop->setDisabled(true);
ui->pushButton_sim->setEnabled(true);
ui->pushButton_simsetup->setEnabled(true);
ui->pushButton_devsetup->setEnabled(true);
ui->pushButton_radsetup->setEnabled(true);
ui->textMessageBox->append("Simulation complete.");
disconnect(wrapper, SIGNAL(thread_finished()),this,SLOT(wrapperDone()));
//wrapper->exit();


}


MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::MainWindow)
{
    ui->setupUi(this);

    //initialize strings
    title_plot="";
    last_path_ioniz="/";

    //initialize buttons, enabled functions, text lines
    ui->pushButton_devsetup->setEnabled(true);
    ui->pushButton_simsetup->setEnabled(false);
    ui->pushButton_radsetup->setEnabled(false);
    this->ui->progressBar->setValue(0);
//    this->ui->horizontalSlider->setValue(slide_generation_point);
    ui->horizontalSlider->setDisabled(true);
    if(this->ui->horizontalSlider->value()==-1){ this->ui->textBrowser->setText("All data");}
    else{this->ui->textBrowser->setText("x=" + QString::number(slide_generation_point)+" um");}
    ui->checkBox->setChecked(false);
    ui->time_spinBox->setDisabled(true);
    ui->pushButton_simsetup->setDisabled(true);
    ui->pushButton_sim->setDisabled(true);
    ui->pushButton_stop->setDisabled(true);
    ui->pushButton_resume->setDisabled(true);

    //initialize device setup variables
    deltax=0;
    deltatp=0;
    deltatn=0;
    resume=0;

    //initialize control variables
    checkdialog=0;
    set_legend=0;
    radiation_control=0;
    lat_front=0;
    control_vac=1;
    control_ion=1;

    //initialize lateral ibic profile plotting values
    gnearest=-1;
    slide_generation_point=-1;
    gpoint=-1;
    xgpoint.resize(2);
    ygpoint.resize(2);
    xgpoint[0]=gpoint;
    xgpoint[1]=gpoint;
    ygpoint[0]=0;
    ygpoint[1]=2;


    //initialize simulation setup values
    lat_front=0;
    n_point = 0;
    n_ion = 0;
    n_eh = 0;
    dispersion = 0;     //um
    e_noise= 0;         // %
    e_threshold= 0;
    x_ioniz.resize(100);
    y_ioniz.resize(100);

    //initialize radiation damage setup
    radhard_p=0.2;
    radhard_n=0.2;
    fluence=0;
    nsigma=1e-19;   //m2
    psigma=1e-19;   //,2
    ptherm=1e5;     //m/s
    ntherm=1e5;     //m/s
    dead_layer=0;
    x_vac.resize(100);
    y_vac.resize(100);

    //initialize random generation number
    init_genrand(time(NULL));

    //launch thread idle
wrapper->start();
}

MainWindow::~MainWindow()
{
    delete ui;
}


void MainWindow::slideChanged(int value)
{
    //slider control of generation position value in lateral ibic plot
    slide_generation_point=value;
}


void MainWindow::on_pushButton_simsetup_clicked()
{
    //launch simulation setup dialog
    Dialog_setup dialog;
    dialog.exec();
    ui->pushButton_resume->setDisabled(true);

    //check dialog exit status and update device properties values
    if(dialog.simok==0)
    {

    lat_front=dialog.lat_front;
    n_point = dialog.n_point;
    n_ion = dialog.n_ion;
    n_eh = dialog.n_eh;
    dispersion = dialog.dispersion*1e-6;    //um
    e_noise= dialog.e_noise/100;            // %
    e_threshold= dialog.e_threshold/100;    // %
    dead_layer=dialog.dead_layer;

    //display current values on text lines
    ui->textMessageBox->append("Number of generation points for simulation: " + QString::number(n_point) + ".");
    ui->textMessageBox->append("Number of ions per point: " + QString::number(n_ion) + " .");
    ui->textMessageBox->append("Number of e-h pairs per ion: " + QString::number(n_eh) + " .");
    ui->textMessageBox->append("Lateral dispersion: " + QString::number(dispersion) + " m.");
    ui->textMessageBox->append("Electronic noise: " + QString::number(e_noise*100) + " % CCE.");
    ui->textMessageBox->append("Noise threshold: " + QString::number(e_threshold*100) + " % CCE.");

   //evaluate number of steps, set up lateral ibic plot, enable simulation and radiation hardness dialogs
   steps = L / n_point;
   ui->horizontalSlider->setSingleStep((int) steps);
   ui->horizontalSlider->setMaximum(steps*(n_point-1));
   ui->pushButton_sim->setEnabled(true);
   ui->pushButton_radsetup->setEnabled(true);


    //import and interpolate ionization profiles in frontal ibic
    if(lat_front==1)
    {
    if(dialog.control_ion==0)
    {
    path_ioniz=dialog.path_ioniz;
    last_path_ioniz=dialog.path_ioniz;
   // last_path_ioniz=last_path_ioniz;

    x_ioniz=dialog.x_ioniz;
    y_ioniz=dialog.y_ioniz;
    yioniz_inter.resize(L);

    for(int i=0; i<L; i++)
    {
    yioniz_inter[i]=0;
    }

    int i=0;
    for(i=0; i<L; i++)
    {
        int j=0;
        while(i*deltax*1e6 > x_ioniz[j] && j< x_ioniz.size())
        {
            j++;
        }

        if(j==x_ioniz.size()-1)
        {
            yioniz_inter[i]=0;
            break;
        }

        if(j==0)
        {
            yioniz_inter[i]=y_ioniz[j];
        }

        else if((i*deltax*1e6-x_ioniz[j-1])<(x_ioniz[j]-i*deltax*1e6))
        {
            yioniz_inter[i]=y_ioniz[j-1];

        }
        else
        {
            yioniz_inter[i]=y_ioniz[j];
        }
    }

    while(i<L-1)
    {
        yioniz_inter[i]=0;
        i++;
    }


    //build cumulative ionization distribution from interpolated profile
    CDF_ion.resize(yioniz_inter.size());
    double sum=0;
    for(int k=0; k<yioniz_inter.size(); k++)
    {
        sum+=yioniz_inter[k];
        CDF_ion[k]=sum;
    }

     for(int k=0; k<CDF_ion.size(); k++)
     {
         CDF_ion[k]/=sum;
     }
    }

    //plot ionization and CDF profiles
    setupIoniz(ui->ioniz_plot);
    ui->ioniz_plot->replot();
    ui->CDF_plot->setVisible(true);
    setupCDF(ui->CDF_plot);
    ui->CDF_plot->replot();
    }

    else
    {
        ui->horizontalSlider->setDisabled(true);
    }

    }

    else
    {
        ui->pushButton_radsetup->setEnabled(false);

    }

    //print simulation parameters and status on text line
    text_simulation();
}



void MainWindow::setupGunn(QCustomPlot *potPlot)
{
    //setup weighting potential from imported files
    potPlot->clearItems();
    //potPlot->plotLayout()->insertRow(0);
    potPlot->plotLayout()->addElement(0, 0, new QCPPlotTitle(potPlot, title_plot));

    potPlot->addGraph();
    potPlot->graph(0)->setPen(QPen(Qt::green));
    potPlot->graph(0)->setBrush(QBrush(QColor(0, 250, 0)));
    potPlot->graph(0)->setAntialiasedFill(false);
    potPlot->graph(0)->setPen(QPen(Qt::blue));

    //rescale to um units
    QVector<double> x(gunn.size()), y0(gunn.size());
    for (int i=0; i<gunn.size(); ++i)
    {
    x[i] = i*deltax/1e-6;
    y0[i] = fabs(gunn[i]);
    }

  potPlot->xAxis->setVisible(true);
  potPlot->xAxis2->setVisible(false);
  potPlot->xAxis2->setTickLabels(true);
  potPlot->yAxis2->setVisible(true);
  potPlot->yAxis2->setTickLabels(false);
  potPlot->graph(0)->setData(x, y0);
  potPlot->graph(0)->rescaleAxes(true);
  potPlot->xAxis->setLabel("Position (um)");
  potPlot->yAxis->setLabel("Weighting potential");
  potPlot->yAxis->setRange(-0.0,1.1);
  potPlot->yAxis->setAutoTickCount(5);
  potPlot->xAxis->setRange(-0.0,L*deltax*1e6);
}

void MainWindow::setupPdrift(QCustomPlot *potPlot)
{
     //setup hole drift velocity probability from imported files
    //potPlot->plotLayout()->insertRow(0);
    potPlot->plotLayout()->addElement(0, 0, new QCPPlotTitle(potPlot, title_plot));

    potPlot->addGraph();
    potPlot->graph(0)->setPen(QPen(Qt::blue));
    potPlot->graph(0)->setBrush(QBrush(QColor(0, 0, 250)));
    potPlot->graph(0)->setAntialiasedFill(false);
    potPlot->graph(0)->setPen(QPen(Qt::blue));

    //rescale abscissa to um
    QVector<double> x(pdrift.size()), y0(pdrift.size());
    for (int i=0; i<pdrift.size(); ++i)
    {
    x[i] = i*deltax/1e-6;
    y0[i] = pdrift[i];
    }

  potPlot->addGraph();
  potPlot->addGraph();
  potPlot->addGraph();
  potPlot->graph(1)->setPen(QPen(Qt::red));
  potPlot->graph(2)->setPen(QPen(Qt::red));
  potPlot->graph(3)->setPen(QPen(Qt::red));

  //plot horizontal lines for 0,0.5,1 probability
  QVector<double> x1(2), y1(2), y2(2), y3(2);
  x1[0]=0;
  x1[1]=gunn.size()*deltax/1e-6;
  y1[0]=0.5;
  y1[1]=0.5;
  y2[0]=1.;
  y2[1]=1.;
  y3[0]=0;
  y3[1]=0;

  potPlot->xAxis->setVisible(true);
  potPlot->xAxis2->setVisible(false);
  potPlot->xAxis2->setTickLabels(true);
  potPlot->yAxis2->setVisible(true);
  potPlot->yAxis2->setTickLabels(false);
  potPlot->yAxis->setTickStep(10);
  potPlot->graph(0)->setData(x, y0);
  potPlot->graph(1)->setData(x1, y1);
  potPlot->graph(2)->setData(x1, y2);
  potPlot->graph(3)->setData(x1, y3);
  potPlot->graph(0)->rescaleAxes(true);
  potPlot->xAxis->setLabel("Position (um)");
  potPlot->yAxis->setLabel("Hole motion probab.");
  potPlot->yAxis->setRange(-0.5,1.5);
  potPlot->xAxis->setRange(-0.0,L*deltax*1e6);
}


void MainWindow::setupNdrift(QCustomPlot *potPlot)
{
    //setup electron drift velocity probability from imported files
    //potPlot->plotLayout()->insertRow(0);
    potPlot->plotLayout()->addElement(0, 0, new QCPPlotTitle(potPlot, title_plot));

    potPlot->addGraph();
    potPlot->graph(0)->setPen(QPen(Qt::blue));
    potPlot->graph(0)->setBrush(QBrush(QColor(0, 0, 250)));
    potPlot->graph(0)->setAntialiasedFill(false);
    potPlot->graph(0)->setPen(QPen(Qt::blue));

    //rescale abscissa to um
    QVector<double> x(ndrift.size()), y0(ndrift.size());

    for (int i=0; i<ndrift.size(); ++i)
    {
    x[i] = i*deltax/1e-6;
    y0[i] = ndrift[i];
    }

  potPlot->addGraph();
  potPlot->addGraph();
  potPlot->addGraph();
  potPlot->graph(1)->setPen(QPen(Qt::red));
  potPlot->graph(2)->setPen(QPen(Qt::red));
  potPlot->graph(3)->setPen(QPen(Qt::red));

  //plot horizontal lines for 0,0.5,1 probability
  QVector<double> x1(2), y1(2), y2(2), y3(2);
  x1[0]=0;
  x1[1]=gunn.size()*deltax/1e-6;
  y1[0]=0.5;
  y1[1]=0.5;
  y2[0]=1.;
  y2[1]=1.;
  y3[0]=0;
  y3[1]=0;

  potPlot->xAxis->setVisible(true);
  potPlot->xAxis2->setVisible(false);
  potPlot->xAxis2->setTickLabels(true);
  potPlot->yAxis2->setVisible(true);
  potPlot->yAxis2->setTickLabels(false);
  potPlot->yAxis->setTickStep(10);
  potPlot->graph(0)->setData(x, y0);
  potPlot->graph(1)->setData(x1, y1);
  potPlot->graph(2)->setData(x1, y2);
  potPlot->graph(3)->setData(x1, y3);
  potPlot->graph(0)->rescaleAxes(true);
  potPlot->xAxis->setLabel("Position (um)");
  potPlot->yAxis->setLabel("Electron motion probab.");
  potPlot->yAxis->setRange(-0.5,1.5);
  potPlot->xAxis->setRange(-0.0,L*deltax*1e6);
}

void MainWindow::setupEfield(QCustomPlot *potPlot)
{
    //setup electric field from imported files
    potPlot->clearItems();
    //potPlot->plotLayout()->insertRow(0);
    potPlot->plotLayout()->addElement(0, 0, new QCPPlotTitle(potPlot, title_plot));

    potPlot->addGraph();
    potPlot->graph(0)->setPen(QPen(Qt::green));
    potPlot->graph(0)->setBrush(QBrush(QColor(0, 250, 0)));
    potPlot->graph(0)->setAntialiasedFill(false);
    potPlot->graph(0)->setPen(QPen(Qt::blue));

    //rescale to (um, V/cm) units
    QVector<double> x(efield.size()), y0(efield.size());
    double emax=0;
    double emin=0;
    //define max range
    for (int i=0; i<efield.size(); ++i)
    {
        x[i] = i*deltax/1e-6;
        y0[i] = efield[i]/100; // V/cm
        if(y0[i]>emax){emax=y0[i];}
        if(y0[i]<emin){emin=y0[i];}
    }

  potPlot->xAxis->setVisible(true);
  potPlot->xAxis2->setVisible(false);
  potPlot->xAxis2->setTickLabels(true);
  potPlot->yAxis2->setVisible(true);
  potPlot->yAxis2->setTickLabels(false);
  potPlot->graph(0)->setData(x, y0);
  potPlot->xAxis->setRange(-0.0,L*deltax*1e6);
  potPlot->yAxis->setRange(emin*1.2,emax*1.2);
  potPlot->xAxis->setLabel("Position (um)");
  potPlot->yAxis->setLabel("Electric field (V/cm)");
}


void MainWindow::setupIoniz(QCustomPlot *potPlot)
{
    //setup ionization profile plot from imported files

    potPlot->addGraph();
    potPlot->graph(0)->setPen(QPen(Qt::green));
    potPlot->graph(0)->setBrush(QBrush(QColor(128, 128, 128)));
    potPlot->graph(0)->setAntialiasedFill(false);
    potPlot->graph(0)->setPen(QPen(Qt::blue));
    potPlot->addGraph();
    potPlot->graph(1)->setPen(QPen(Qt::red));
    potPlot->graph(1)->setBrush(QBrush(QColor(250, 0, 0)));
    potPlot->graph(1)->setAntialiasedFill(false);

    //convert to (um, eV/cm) units imported 100 array (ioniz_um) and interpolated array (ioniz_input)
    QVector<double> x_ioniz_um, y_ioniz_um,y_ioniz_input;
    x_ioniz_um.resize(x_ioniz.size());
    y_ioniz_um.resize(y_ioniz.size());
    y_ioniz_input.resize(yioniz_inter.size());
    double imax;
    imax=0;

    for(int i=0; i<x_ioniz.size(); i++)
    {
    x_ioniz_um[i]=x_ioniz[i];
    y_ioniz_um[i]=y_ioniz[i]/100;
    }

    for(int i=0; i< yioniz_inter.size();i++)
    {
    y_ioniz_input[i]=yioniz_inter[i]/100;
    }

    QVector<double> x_array_interpol;
    x_array_interpol.resize(L);

    for(int i=0; i<L; i++)
    {
    x_array_interpol[i]=i*deltax*1e6;
    }

  potPlot->xAxis->setVisible(true);
  potPlot->xAxis2->setVisible(false);
  potPlot->xAxis2->setTickLabels(true);
  potPlot->yAxis2->setVisible(true);
  potPlot->yAxis2->setTickLabels(false);
 // potPlot->yAxis->setTickStep(10);

  //plot imported file and interpolation profile
  potPlot->graph(0)->setData(x_ioniz_um, y_ioniz_um );
  potPlot->graph(1)->setData(x_array_interpol, y_ioniz_input);
  potPlot->xAxis->setLabel("Position (um)");
  potPlot->yAxis->setLabel("Ionization (eV/cm)");
  potPlot->graph(0)->rescaleAxes(true);
  potPlot->graph(1)->rescaleAxes(true);
  potPlot->xAxis->setRange(0.0,x_ioniz[x_ioniz.size()-1]);
}


void MainWindow::setupCDF(QCustomPlot *potPlot)
{
    //setup CDF profile plot
    potPlot->addGraph();
    potPlot->graph(0)->setPen(QPen(Qt::green));
    potPlot->graph(0)->setBrush(QBrush(QColor(128, 128, 128)));
    potPlot->graph(0)->setAntialiasedFill(false);
    potPlot->graph(0)->setPen(QPen(Qt::blue));

    //rescale to (um, [ ])

    QVector<double> x_CDF_um;
    x_CDF_um.resize(CDF_ion.size());
    for(int i=0; i< CDF_ion.size(); i++)
    {
        x_CDF_um[i]=i*deltax*1e6;
    }

  potPlot->xAxis->setVisible(true);
  potPlot->xAxis2->setVisible(false);
  potPlot->xAxis2->setTickLabels(true);
  potPlot->yAxis2->setVisible(true);
  potPlot->yAxis2->setTickLabels(false);
  potPlot->graph(0)->setData(x_CDF_um, CDF_ion );
  potPlot->xAxis->setLabel("Position (um)");
  potPlot->yAxis->setLabel("Ionization cumulative distribution function");
  potPlot->graph(0)->rescaleAxes(true);
  potPlot->xAxis->setRange(0.0,x_ioniz[x_ioniz.size()-1]);
  potPlot->yAxis->setRange(0.0,1.00);
}

void MainWindow::setupVac(QCustomPlot *potPlot)
{
    //setup vacancy density profile plot
    potPlot->addGraph();
    potPlot->graph(0)->setPen(QPen(Qt::green));
    potPlot->graph(0)->setBrush(QBrush(QColor(0, 250, 250)));
    potPlot->graph(0)->setAntialiasedFill(false);
    potPlot->graph(0)->setPen(QPen(Qt::blue));
    potPlot->addGraph();
    potPlot->graph(1)->setPen(QPen(Qt::black));
    potPlot->graph(1)->setBrush(QBrush(QColor(250, 0, 0)));
    potPlot->graph(1)->setAntialiasedFill(false);


    //rescale to (um, vac/cm) imported file (vac) and interpolated file (array_interpol)
    QVector<double> x_vac_um, y_vac_um;
    x_vac_um.resize(x_vac.size());
    y_vac_um.resize(y_vac.size());
    for(int i=0; i<x_vac.size(); i++)
    {
        x_vac_um[i]=x_vac[i]; //in unità di micrometro
        y_vac_um[i]=y_vac[i]/100; //in unità vac/cm
    }


    QVector<double> x_array_interpol, y_array_interpol;
    x_array_interpol.resize(L);
    y_array_interpol.resize(L);

    for(int i=0; i<L; i++)
    {
        x_array_interpol[i]=i*deltax*1e6;
        y_array_interpol[i]=yvac_inter[i]/100;
    }

  potPlot->xAxis->setVisible(true);
  potPlot->xAxis2->setVisible(false);
  potPlot->xAxis2->setTickLabels(true);
  potPlot->yAxis2->setVisible(true);
  potPlot->yAxis2->setTickLabels(false);
  potPlot->graph(0)->setData(x_vac_um, y_vac_um);
  potPlot->graph(1)->setData(x_array_interpol, y_array_interpol);
  potPlot->xAxis->setLabel("Position (um)");
  potPlot->yAxis->setLabel("Vacancy dens. (1/cm)");
  potPlot->graph(0)->rescaleAxes(true);
  potPlot->xAxis->setRange(0.0,x_vac[x_vac.size()-1]);

  //define y-axis max value
  double max_vac=0;

  for(int i=0; i<y_vac_um.size(); i++)
  {
      if(max_vac<y_vac_um[i]) max_vac=y_vac_um[i];
  }
  potPlot->yAxis->setRange(0.0,max_vac*1.2);
}


void MainWindow::setupTau(QCustomPlot *potPlot)
{
    //setup lifetime plot
    potPlot->addGraph();
    potPlot->graph(0)->setPen(QPen(Qt::blue));
    potPlot->graph(0)->setAntialiasedFill(false);
    potPlot->addGraph();
    potPlot->graph(1)->setPen(QPen(Qt::red));
    potPlot->graph(1)->setAntialiasedFill(false);

    //define interpolated arrays for x (um), taup (red) and taun (blue) (ns)
    QVector<double> x_array_interpol,taun_interpol, taup_interpol;
    x_array_interpol.resize(L);
    taun_interpol.resize(L);
    taup_interpol.resize(L);

    for(int i=0; i<L; i++)
    {
        x_array_interpol[i]=i*deltax*1e6;
        taun_interpol[i]=taun_arr[i]*1e9;
        taup_interpol[i]=taup_arr[i]*1e9;
    }

  potPlot->xAxis->setVisible(true);
  potPlot->xAxis2->setVisible(false);
  potPlot->xAxis2->setTickLabels(true);
  potPlot->yAxis2->setVisible(true);
  potPlot->yAxis2->setTickLabels(false);
  potPlot->graph(0)->setData(x_array_interpol, taun_interpol);
  potPlot->graph(1)->setData(x_array_interpol, taup_interpol);
  potPlot->xAxis->setLabel("Position (um)");
  potPlot->yAxis->setLabel("Lifetime (ns)");

  //maximum is given by tauplot_scale_x/y values
  potPlot->xAxis->setRange(0.0,tauplot_scale_x);
  potPlot->yAxis->setRange(0.0,tauplot_scale_y);
}


void MainWindow::legendClick()
{
    //set legend visible when clicked on lateral ibic profile plot
    ui->mc_plot->legend->setVisible(false);
    ui->mc_plot->replot();
}

void MainWindow::setupMC(QCustomPlot *potPlot, QCustomPlot *histo, QCustomPlot *time)
{
    //setup monte carlo simulation and simulation plots
    //histo plot: simulation spectra. graph 0: counts, blue; graph 1: mean value, red line
    //time plot: time resolved CCE
    //potPlot: lateral ibic profile

    histo->addGraph();
    histo->graph(0)->setPen(QPen(Qt::black));
    histo->graph(0)->setLineStyle(QCPGraph::lsStepCenter);
    histo->graph(0)->setBrush(QBrush(QColor(0, 0, 250)));
    histo->graph(0)->setAntialiasedFill(false);
    histo->xAxis->setVisible(true);
    histo->xAxis2->setVisible(false);
    histo->xAxis2->setTickLabels(true);
    histo->yAxis2->setVisible(true);
    histo->yAxis2->setTickLabels(false);
    histo->graph(0)->rescaleAxes(true);
    histo->xAxis->setTickStep(0.25);
    histo->xAxis2->setTickStep(0.25);
    histo->xAxis->setAutoTickCount(5);
    histo->xAxis->setRange(0.0,1.1);
    histo->xAxis->setLabel("CCE");
    histo->yAxis->setLabel("counts");
    histo->addGraph();
    QPen graphPen_red;
    graphPen_red.setColor(Qt::red);
    graphPen_red.setWidthF(2);
    histo->graph(1)->setPen(graphPen_red);
    //histo->plotLayout()->insertRow(0);
    histo->plotLayout()->addElement(0, 0, new QCPPlotTitle(histo, title_plot));

    QCPItemText *textLabel = new QCPItemText(histo);
    histo->addItem(textLabel);
    textLabel->position->setType(QCPItemPosition::ptAxisRectRatio);
    textLabel->position->setCoords(0.5,0);
    textLabel->setText("Mean: " + QString::number(ion));
    textLabel->setPen(QPen(Qt::black));
    textLabel->setPositionAlignment(Qt::AlignTop|Qt::AlignHCenter);

    //plot mean time-resolved CCE
    time->addGraph();
    time->graph(0)->setPen(QPen(Qt::red));
    time->xAxis->setVisible(true);
    time->xAxis2->setVisible(false);
    time->xAxis2->setTickLabels(true);
    time->yAxis2->setVisible(true);
    time->yAxis2->setTickLabels(false);
    time->xAxis->setAutoTickCount(5);
    //time->plotLayout()->insertRow(0);
    time->plotLayout()->addElement(0, 0, new QCPPlotTitle(time, title_plot));

    time->xAxis->setLabel("Time (ns)");
    time->yAxis->setLabel("CCE");

    //plot profiles: pulses (gray scatter), mean (red), median (black)
    potPlot->addGraph();
    potPlot->graph(0)->setPen(QPen(Qt::gray));
    potPlot->graph(0)->setLineStyle(QCPGraph::lsNone);
    potPlot->graph(0)->setScatterStyle(QCPScatterStyle(QCPScatterStyle::ssCircle, 2));

  //  potPlot->graph(0)->setScatterSize(2);
    potPlot->graph(0)->setBrush(QBrush(QColor(0, 250, 0)));
    potPlot->graph(0)->setAntialiasedFill(false);
    potPlot->graph(0)->setName("Simulated pulses");
    //potPlot->plotLayout()->insertRow(0);
    potPlot->plotLayout()->addElement(0, 0, new QCPPlotTitle(potPlot, title_plot));


    potPlot->addGraph();
    potPlot->graph(1)->setPen(QPen(Qt::red));
    potPlot->graph(1)->setLineStyle(QCPGraph::lsLine);
    potPlot->graph(1)->setName("Mean profile");

    potPlot->addGraph();
    potPlot->graph(2)->setPen(QPen(Qt::black));
    potPlot->graph(2)->setLineStyle(QCPGraph::lsLine);
    potPlot->graph(2)->setName("Median profile");

    potPlot->xAxis->setVisible(true);
    potPlot->xAxis2->setVisible(false);
    potPlot->xAxis2->setTickLabels(true);
    potPlot->yAxis2->setVisible(true);
    potPlot->yAxis2->setTickLabels(false);
    potPlot->graph(0)->rescaleAxes(true);
    potPlot->yAxis->setRange(-0.,1.2);
    potPlot->xAxis->setRange(-0.0,L*deltax*1e6);
    potPlot->xAxis->setLabel("Position (um)");
    potPlot->yAxis->setLabel("CCE");

    //plot experimental profile (if selected): out of date feature
    potPlot->addGraph();
    QPen graphPen;
    graphPen.setColor(Qt::darkCyan);
    graphPen.setWidthF(1);
    potPlot->graph()->setPen(graphPen);
    potPlot->graph(3)->setLineStyle(QCPGraph::lsLine);
    potPlot->graph(3)->removeFromLegend();

    //add graphs to highlight geeneration position from horizontal slider selection
    potPlot->addGraph();
    potPlot->addGraph();
    potPlot->graph(4)->removeFromLegend();
    potPlot->graph(5)->removeFromLegend();

    if(set_legend==0)   { potPlot->legend->setVisible(false);}
    if(set_legend==1)   { potPlot->legend->setVisible(true);}
    potPlot->legend->setMaximumSize(50,30);
    potPlot->legend->setMinimumSize(50,30);
    QFont legendFont = font();  // start out with MainWindow's font..
    legendFont.setPointSize(9); // and make a bit smaller for legend
    potPlot->legend->setFont(legendFont);
    potPlot->axisRect()->insetLayout()->setInsetAlignment(0, Qt::AlignTop|Qt::AlignRight);


    //setup thread for MC simulation
    //if resume is not activated, setup from scratch copying mainwindow values to thread memory
    if(resume==0)
    {

  wrapper->lat_front=lat_front;
  wrapper-> temp = temp;
  wrapper-> L =L;
  wrapper-> ndiff = ndiff;
  wrapper-> pdiff = pdiff;
  wrapper-> gunn = gunn;
  wrapper-> n_point = n_point;
  wrapper-> n_ion=n_ion;
  wrapper-> n_eh = n_eh;
  wrapper-> dispersion = dispersion;
  wrapper-> e_noise = e_noise;
  wrapper-> e_threshold = e_threshold;
  wrapper-> ion = ion;
  wrapper-> p = p;
  wrapper-> x0=x0;
  wrapper-> steps=steps;
  wrapper-> taup = taup;
  wrapper-> taun = taun;
  wrapper-> deltax =deltax;
  wrapper-> deltatn = deltatn;
  wrapper-> deltatp = deltatp;
  wrapper-> progress = progress;
  wrapper-> end_sim = end_sim;
  wrapper->cce.resize(cce.size());
  wrapper-> cce = cce;
  wrapper-> cce_noise.resize(cce_noise.size());
  wrapper-> cce_noise = cce_noise;
  wrapper->cce_threshold.resize(cce_threshold.size());
  wrapper-> cce_threshold = cce_threshold;
  wrapper->multipos.resize(multipos.size());
  wrapper-> multipos = multipos;
  wrapper->cce_median.resize(cce_median.size());
  wrapper-> cce_median = cce_median;
  wrapper->tot_spectrum.resize(tot_spectrum.size());
  wrapper-> tot_spectrum = tot_spectrum;
  wrapper->maximum.resize(maximum.size());
  wrapper-> maximum = maximum;

  wrapper-> gnearest = gnearest;
  wrapper-> tot_hit_over = tot_hit_over;
  wrapper-> tot_mean_plot = tot_mean_plot;
  wrapper->histo_step.resize(histo_step.size());
  wrapper-> histo_step = histo_step;
  wrapper->x_vac.resize(x_vac.size());
  wrapper-> x_vac = x_vac;
  wrapper->y_vac.resize(y_vac.size());
  wrapper-> y_vac = y_vac;
  wrapper->x_ioniz.resize(x_ioniz.size());
  wrapper-> x_ioniz = x_ioniz;
  wrapper->efield.resize(efield.size());
  wrapper-> efield = efield;
  wrapper->pdrift.resize(pdrift.size());
  wrapper-> pdrift = pdrift;
   wrapper->ndrift.resize(ndrift.size());
  wrapper-> ndrift = ndrift;
  wrapper->taup_arr.resize(taup_arr.size());
  wrapper-> taup_arr = taup_arr;
  wrapper->taun_arr.resize(taun_arr.size());
  wrapper-> taun_arr = taun_arr;
  wrapper->ptaun.resize(ptaun.size());
  wrapper-> ptaun = ptaun;
   wrapper->ptaup.resize(ptaup.size());
  wrapper-> ptaup = ptaup;
  wrapper->yioniz_inter.resize(yioniz_inter.size());
  wrapper-> yioniz_inter = yioniz_inter;
  wrapper->CDF_ion.resize(CDF_ion.size());
  wrapper-> CDF_ion = CDF_ion;
  wrapper->cce_frontal.resize(cce_frontal.size());
  wrapper-> cce_frontal = cce_frontal;
  wrapper->cce_frontal_ion_n.resize(cce_frontal_ion_n.size());
  wrapper-> cce_frontal_ion_n = cce_frontal_ion_n;
  wrapper->cce_frontal_ion_p.resize(cce_frontal_ion_p.size());
  wrapper-> cce_frontal_ion_p = cce_frontal_ion_p;
  wrapper->cce_frontal_thresh.resize(cce_frontal_thresh.size());
  wrapper-> cce_frontal_thresh = cce_frontal_thresh;
  wrapper->cce_time_frontal_tot.resize(cce_time_frontal_tot.size());
  wrapper-> cce_time_frontal_tot = cce_time_frontal_tot;
  wrapper->cce_tfrontal.resize(cce_tfrontal.size());
  wrapper-> cce_tfrontal = cce_tfrontal;
  wrapper->cce_tfrontal_aux.resize(cce_tfrontal_aux.size());
  wrapper-> cce_tfrontal_aux = cce_tfrontal_aux;

  wrapper-> hit_frontal = hit_frontal;
  wrapper-> aux_mean_frontal = aux_mean_frontal;
  wrapper->cce_time_frontal_p.resize(cce_time_frontal_p.size());
  wrapper-> cce_time_frontal_p = cce_time_frontal_p;
  wrapper->cce_time_frontal_n.resize(cce_time_frontal_n.size());
  wrapper-> cce_time_frontal_n = cce_time_frontal_n;
  wrapper->xgpoint.resize(xgpoint.size());
  wrapper-> xgpoint = xgpoint;
  wrapper->ygpoint.resize(ygpoint.size());
  wrapper-> ygpoint = ygpoint;
  wrapper-> slide_generation_point = slide_generation_point;
  wrapper-> gpoint = gpoint;
  wrapper->cce_frontal_ion_p2.resize(cce_frontal_ion_p2.size());
  wrapper-> cce_frontal_ion_p2 = cce_frontal_ion_p2;
  wrapper->cce_frontal_ion_n2.resize(cce_frontal_ion_n2.size());
  wrapper-> cce_frontal_ion_n2 = cce_frontal_ion_n2;
  wrapper->cce_tfrontal2.resize(cce_tfrontal2.size());
  wrapper->cce_tfrontal2 = cce_tfrontal2;
  wrapper->cce_tfrontal_aux2.resize(cce_tfrontal_aux2.size());
  wrapper-> cce_tfrontal_aux2 = cce_tfrontal_aux2;
  wrapper->cce_time_frontal_p2.resize(cce_time_frontal_p2.size());
  wrapper->cce_time_frontal_p2 = cce_time_frontal_p2;
  wrapper->cce_time_frontal_n2.resize(cce_time_frontal_n2.size());
  wrapper-> cce_time_frontal_n2 = cce_time_frontal_n2;
  wrapper-> maxtscale = maxtscale;
  wrapper-> valtscale = valtscale;
  wrapper->prog_show=0;
  wrapper->num_t_plot=num_t_plot;
  wrapper->hit_over_threshold.resize(hit_over_threshold.size());
  wrapper->hit_over_threshold=hit_over_threshold;
  wrapper->cce_mean.resize(cce_mean.size());
  wrapper->cce_mean=cce_mean;
  wrapper->x_histo_mean.resize(x_histo_mean.size());
  wrapper->x_histo_mean=x_histo_mean;
  wrapper->y_histo_mean.resize(y_histo_mean.size());
  wrapper->y_histo_mean=y_histo_mean;
  wrapper->cce_mean_plot.resize(cce_mean_plot.size());
  wrapper->cce_mean_plot=cce_mean_plot;

  }

  //if resume is selected, keep current thread values and deactivate resume mode for next thread initialization
  if(resume==1)
  {
      resume=0;
  }

  //connect thread functions with progress bar, plot update function
  connect(wrapper, SIGNAL(updateProgressbar(int)),ui->progressBar, SLOT(setValue(int)));
  connect(wrapper, SIGNAL(passValues()), SLOT(updatePlot() ) );
  connect(this, SIGNAL(thread()), SLOT(updatePlot() ) );

  //connect thread end signal with thread termination function
  connect(wrapper, SIGNAL(thread_finished()),this,SLOT(wrapperDone()));
 // connect(wrapper, SIGNAL(finished()), wrapper, SLOT(deleteLater()));
 //segnalano ancora questo! connect(thread, SIGNAL(finished()), worker, SLOT(deleteLater()));
}




void MainWindow::updatePlot( )
{

   qDebug()<<"entered function updatePlot";
            cce_frontal_thresh=wrapper->cce_frontal_thresh;
            tot_mean_plot=wrapper->tot_mean_plot;
            cce_time_frontal_tot=wrapper->cce_time_frontal_tot;
            cce_tfrontal2=wrapper-> cce_tfrontal2;
            tot_hit_over=wrapper-> tot_hit_over;
            cce_median=wrapper-> cce_median;
            cce_threshold=wrapper-> cce_threshold;
            multipos=wrapper-> multipos;
            maximum=wrapper-> maximum;
            histo_step=wrapper-> histo_step;
            tot_spectrum=wrapper-> tot_spectrum;
            cce_tfrontal=wrapper->cce_tfrontal;
            cce_mean_frontal=wrapper->cce_mean_frontal;
            x_histo_mean=wrapper->x_histo_mean;
            y_histo_mean=wrapper->y_histo_mean;
            hit_over_threshold=wrapper->hit_over_threshold;
            cce_mean=wrapper->cce_mean;
            cce_mean_plot=wrapper->cce_mean_plot;

            qDebug() << "aggiornato valori scalari";
            qDebug() << "lat_front: "<< lat_front;

            if(lat_front==1){


             ui->time_plot->graph(0)->clearData();
             ui->time_plot->graph(0)->setData(cce_time_frontal_tot,cce_tfrontal);


             ui->time_plot->xAxis->setRange(0,valtscale);
             ui->time_plot->yAxis->setRange(0,1);
             ui->time_plot->replot();


             ui->histo_plot->graph(0)->clearData();
             ui->histo_plot->graph(1)->clearData();
             ui->histo_plot->graph(0)->addData(histo_step, tot_spectrum[0]);
             ui->histo_plot->graph(1)->addData(x_histo_mean, y_histo_mean);

             ui->histo_plot->clearItems();
             QCPItemText *textLabel = new QCPItemText(ui->histo_plot);
             ui->histo_plot->addItem(textLabel);
             textLabel->position->setType(QCPItemPosition::ptAxisRectRatio);
             textLabel->position->setCoords(0.5,0);
             textLabel->setText("Mean: " + QString::number(cce_mean_frontal));
             textLabel->setPen(QPen(Qt::black));
             textLabel->setPositionAlignment(Qt::AlignTop|Qt::AlignHCenter);

             ui->histo_plot->yAxis->setRange(-0.0,maximum[0]*1.2);
             ui->histo_plot->replot();


                }


if(lat_front==0){

     qDebug() << "entrato in update_plot laterale";
      ui->mc_plot->graph(3)->clearData();
      ui->mc_plot->graph(4)->clearData();
      ui->mc_plot->graph(3)->setData(xgpoint,ygpoint);

      qDebug() << "pulito grafici";
      ui->time_plot->graph(0)->clearData();
      if(gnearest>=0)
      {
      ui->time_plot->graph(0)->setData(cce_time_frontal_tot,cce_tfrontal2[gnearest]);
      ui->time_plot->xAxis->setRange(0,valtscale);
      ui->time_plot->yAxis->setRange(0,1.1);
      ui->time_plot->replot();
      }

      qDebug() << "aggiornato if(gnearest>0)";

    ui->mc_plot->graph(0)->clearData();
    for(int j=0; j<n_point; j++){
     ui->mc_plot->graph(0)->addData(multipos[j], cce_threshold[j]);
    }
     ui->mc_plot->graph(1)->clearData();
     ui->mc_plot->graph(1)->addData(gen_pos, cce_mean_plot);
     ui->mc_plot->graph(2)->clearData();
     ui->mc_plot->graph(2)->addData(gen_pos, cce_median);

     ui->mc_plot->replot();
     ui->histo_plot->clearItems();
     ui->histo_plot->graph(0)->clearData();
     ui->histo_plot->graph(1)->clearData();


     qDebug() << "finito i cleardata";

     QVector<QVector<double> > x_histo_mean_lateral, y_histo_mean_lateral;
     x_histo_mean_lateral.resize(n_point+1);
     y_histo_mean_lateral.resize(n_point+1);

     x_histo_mean_lateral[0].resize(2);
     y_histo_mean_lateral[0].resize(2);
     if(tot_mean_plot/tot_hit_over>0)
     {
     x_histo_mean_lateral[0][0]=tot_mean_plot/tot_hit_over;
     x_histo_mean_lateral[0][1]=tot_mean_plot/tot_hit_over;
     }
     else
     {
         x_histo_mean_lateral[0][0]=0;
         x_histo_mean_lateral[0][1]=0;
     }
     y_histo_mean_lateral[0][0]=0;
     y_histo_mean_lateral[0][1]=maximum[0];

     qDebug() << "aggiornato histo_mean_lateral1";

     for(int q=1; q<n_point; q++)
     {
         x_histo_mean_lateral[q].resize(2);
         y_histo_mean_lateral[q].resize(2);
         x_histo_mean_lateral[q][0]=cce_mean_plot[q-1];
         x_histo_mean_lateral[q][1]=cce_mean_plot[q-1];
         y_histo_mean_lateral[q][0]=0;
         y_histo_mean_lateral[q][1]=maximum[q];

     }
qDebug() << "aggiornato histo_mean_lateral2";



            if(gnearest<0){
                ui->histo_plot->graph(0)->setData(histo_step,tot_spectrum[0]);
                ui->histo_plot->graph(1)->setData(x_histo_mean_lateral[0], y_histo_mean_lateral[0]);

                 ui->histo_plot->yAxis->setRange(-0.0,maximum[0]*1.2);

                 QCPItemText *textLabel = new QCPItemText(ui->histo_plot);
                 ui->histo_plot->addItem(textLabel);
                 textLabel->position->setType(QCPItemPosition::ptAxisRectRatio);
                 textLabel->position->setCoords(0.5,0);
                 textLabel->setText("Mean: " + QString::number(tot_mean_plot/tot_hit_over));
                 textLabel->setPen(QPen(Qt::black));
                 textLabel->setPositionAlignment(Qt::AlignTop|Qt::AlignHCenter);
            }
            else
            {
                ui->histo_plot->graph(0)->setData(histo_step,tot_spectrum[gnearest+1]);
                ui->histo_plot->graph(1)->setData(x_histo_mean_lateral[gnearest+1], y_histo_mean_lateral[gnearest+1]);

                 ui->histo_plot->yAxis->setRange(-0.0,maximum[gnearest+1]*1.2);
                 QCPItemText *textLabel = new QCPItemText(ui->histo_plot);
                 ui->histo_plot->addItem(textLabel);
                 textLabel->position->setType(QCPItemPosition::ptAxisRectRatio);
                 textLabel->position->setCoords(0.5,0);
                 textLabel->setText("Mean: " + QString::number(cce_mean_plot[gnearest]));
                 textLabel->setPen(QPen(Qt::black));
                 textLabel->setPositionAlignment(Qt::AlignTop|Qt::AlignHCenter);
            }


            qDebug() << "aggiornato if(gnearest<0)";

            ui->histo_plot->replot();

            qDebug() << "riplottato";

            }

}


void MainWindow::on_pushButton_sim_clicked()
{


//MyLibraryWrapper *wrapper = new MyLibraryWrapper();
//wrapper->start();

resume=0;

if(lat_front==0)
{
    ui->horizontalSlider->setEnabled(true);
}
if(lat_front==1)
{
    ui->horizontalSlider->setEnabled(false);
}
    ui->pushButton_devsetup->setDisabled(true);
    ui->pushButton_radsetup->setDisabled(true);
    ui->pushButton_simsetup->setDisabled(true);
    ui->pushButton_sim->setDisabled(true);
    ui->pushButton_stop->setEnabled(true);
    ui->pushButton_resume->setDisabled(true);

    cce.resize(0);
    cce_noise.resize(0);

    gen_pos.resize(n_point);
    cce_mean.resize(n_point);
    cce_mean_plot.resize(n_point);
    hit_over_threshold.resize(n_point);
    cce_threshold.resize(n_point);
    cce_median.resize(n_point);
    multipos.resize(n_point);

    qDebug() << n_point << "=n_point";

    for(int k=0; k<n_point; k++)
    {
        gen_pos[k]=k*steps*deltax*1e6;
        cce_mean[k]=0;
        cce_mean_plot[k]=0;
        cce_median[k]=0;
        hit_over_threshold[k]=0;
        cce_threshold[k].resize(0);
        multipos[k].resize(0);

    }

            tot_spectrum.resize(n_point+1);
            maximum.resize(n_point+1);



    histo_step.resize(110);

    for(int i=0; i<n_point+1; i++)
    {
        tot_spectrum[i].resize(110);
        maximum[i]=0;

    for(int k=0; k<110; k++){
       tot_spectrum[i][k]=0;

    }
    }
    for(int k=0; k<histo_step.size(); k++)
    {

        histo_step[k]=k*0.01;
    }

    ion=0;
    p=0;
    progress=0;
    end_sim=n_ion*n_point;

   cce_frontal.resize(0);
   cce_frontal_thresh.resize(0);
   aux_mean_frontal=0;
   hit_frontal=0;


    qDebug() <<  "'opzioni temporali";

   //opzioni temporali


    //time_resolved: scelgo array da 0 a 3*taumax, a step di 0.1 ns
    double massimo;
    if(taun>taup)
    {
    massimo=3*taun;
    }
    else
    {
    massimo=3*taup;
    }

    maxtscale=massimo*1e9;
    valtscale=massimo/3*1e9;
    ui->time_spinBox->setValue(valtscale);
    ui->time_spinBox->setMaximum(maxtscale);
    ui->time_spinBox->setSingleStep(maxtscale/300);

    ui->time_spinBox->setEnabled(true);

    if(massimo>100*deltatn && 100*deltatp)
    {
    double aux_time=massimo/100;
    num_t_plot=100;

    while (aux_time>100*deltatn && aux_time>100*deltatp)
    {
        num_t_plot+=99;
        aux_time/=100;
    }




   // if(massimo/10;
    //TRIBIC
    // 5e-11 : 21 steps from 0 to 1 ns
   // 1e-9: 99 steps to 100 ns points: simulation at each first 10 ns
   // 1e-8  99 steps to 1000 ns
    //1e-7: 99 steps to 10 us


     qDebug() << n_point << "rescale vari";

   cce_frontal_ion_p.resize(num_t_plot); //decido arbitrariamente 100 punti x time resoluved: check qui in caso cambi!
   wrapper->cce_frontal_ion_p.resize(num_t_plot);

   cce_frontal_ion_n.resize(num_t_plot);
   wrapper->cce_frontal_ion_n.resize(num_t_plot);
   cce_tfrontal.resize(num_t_plot);
   wrapper->cce_tfrontal.resize(num_t_plot);
   cce_tfrontal_aux.resize(num_t_plot);
   wrapper->cce_tfrontal_aux.resize(num_t_plot);
   cce_time_frontal_tot.resize(num_t_plot);
   wrapper->cce_time_frontal_tot.resize(num_t_plot);
   cce_time_frontal_n.resize(num_t_plot);
   wrapper->cce_time_frontal_n.resize(num_t_plot);
   cce_time_frontal_p.resize(num_t_plot);
   wrapper->cce_time_frontal_p.resize(num_t_plot);
   cce_time_frontal_n2.resize(n_point);
   wrapper->cce_time_frontal_n2.resize(n_point);
   cce_time_frontal_p2.resize(n_point);
   wrapper->cce_time_frontal_p2.resize(n_point);
   cce_tfrontal2.resize(n_point);
   wrapper->cce_tfrontal2.resize(n_point);
   cce_tfrontal_aux2.resize(n_point);
   wrapper->cce_tfrontal_aux2.resize(n_point);
   cce_frontal_ion_p2.resize(n_point);
   wrapper->cce_frontal_ion_p2.resize(n_point);
   cce_frontal_ion_n2.resize(n_point);
   wrapper->cce_frontal_ion_n2.resize(n_point);

   for(int i=0; i<n_point; i++)
   {
       cce_time_frontal_n2[i].resize(num_t_plot);
       cce_time_frontal_p2[i].resize(num_t_plot);
       cce_tfrontal2[i].resize(num_t_plot);
       cce_tfrontal_aux2[i].resize(num_t_plot);
       cce_frontal_ion_p2[i].resize(num_t_plot);
       cce_frontal_ion_n2[i].resize(num_t_plot);
       wrapper->cce_time_frontal_n2[i].resize(num_t_plot);
       wrapper->cce_time_frontal_p2[i].resize(num_t_plot);
       wrapper->cce_tfrontal2[i].resize(num_t_plot);
       wrapper->cce_tfrontal_aux2[i].resize(num_t_plot);
       wrapper->cce_frontal_ion_p2[i].resize(num_t_plot);
       wrapper->cce_frontal_ion_n2[i].resize(num_t_plot);
   }

 qDebug() << n_point << "rescale terminati";

   ui->time_spinBox->setEnabled(true);



   cce_time_frontal_p[0]=0;
   cce_time_frontal_n[0]=0;
   cce_time_frontal_tot[0]=0;

    qDebug() << n_point << "time_frontal[0] terminato";


   int j=0;
   aux_time=massimo/100;
   //old setup ...

   for(int i=num_t_plot-1; i>0; i--)
   {
       if(j==99)
       {
           j=0;
           aux_time/=100;
       }

 cce_time_frontal_p[i]=((100-j)*aux_time/deltatp);
 cce_time_frontal_n[i]=((100-j)*aux_time/deltatp);



 cce_tfrontal[i]=0;
 cce_tfrontal_aux[i]=0;
 cce_time_frontal_tot[i]=cce_time_frontal_p[i]*deltatp*1e9; //in unità di ns
 j++;
   }


    qDebug() << n_point << "primo ciclo terminati";

   for(int j=0; j<n_point; j++)
   {
       for(int i=0; i<num_t_plot; i++)
       {

           cce_time_frontal_n2[j][i]=cce_time_frontal_n[i];
           cce_time_frontal_p2[j][i]=cce_time_frontal_p[i];
      cce_tfrontal2[j][i]=0;
      cce_tfrontal_aux2[j][i]=0;
        }
   }

}

 qDebug() << n_point << "pulisco grafici";


ui->histo_plot->clearItems();
ui->mc_plot->clearItems();
  ui->mc_plot->clearGraphs();
  ui->histo_plot->clearGraphs();
  ui->time_plot->clearGraphs();

   qDebug() << n_point << "avvio thread";
  wrapper->lat_front=lat_front;
  wrapper->start_thread();
 setupMC(ui->mc_plot, ui->histo_plot,ui->time_plot);
 ui->mc_plot->replot();
 ui->histo_plot->replot();
 ui->time_plot->replot();



  ui->textMessageBox->append("Simulation started.");
 qDebug() << "Simulation started.";

}

void MainWindow::on_pushButton_resume_clicked()
{
resume=1;
//wrapper->start();

ui->pushButton_stop->setEnabled(true);
ui->pushButton_sim->setDisabled(true);
ui->pushButton_resume->setDisabled(true);
ui->pushButton_simsetup->setDisabled(true);
wrapper->start_thread();
qDebug()<<"comando start thread";

  ui->textMessageBox->append("Simulation resumed.");
  ui->histo_plot->clearItems();
  ui->mc_plot->clearItems();
  ui->mc_plot->clearGraphs();
  setupMC(ui->mc_plot, ui->histo_plot,ui->time_plot);
  ui->mc_plot->replot();
  ui->histo_plot->replot();
  ui->time_plot->replot();


}


void MyLibraryWrapper::frontal()
{
//qDebug()<< "entered frontal wrapper";

    if(thread_status==0)
    {

    if(n_ion>0){

  // calculate two new data points:
#if QT_VERSION < QT_VERSION_CHECK(4, 7, 0)
     double key = 0;
#else
  double key = QDateTime::currentDateTime().toMSecsSinceEpoch()/1000.0;
#endif

  int prog_show = progress/n_ion*100;
  emit updateProgressbar(prog_show);

  static double lastPointKey = 0;
  if (key-lastPointKey >.05) // at most add point every 10 ms
  {

   //   qDebug()<< "ion started";
      if(ion<n_ion)
        {
          double ctot=0;
          double counterp=0;
          double countern=0;
          int x;

   //        qDebug()<< "fill cce_frontal_ion";

          for(int i=0; i<num_t_plot; i++)
          {
              cce_frontal_ion_p[i]=0;
              cce_frontal_ion_n[i]=0;
          }

// qDebug()<< "simulate pairs";

          for(int h=0;h<n_eh;h++)
            {

       //        qDebug()<< "DISTRIBUZIONE CUMULATIVA IN PROFONDITA";
              //DISTRIBUZIONE CUMULATIVA IN PROFONDITA'

                        int index=0;
                        double random2=genrand_real1();
                                while(random2 > CDF_ion[index])
                                {
                                    index++;
                                }
                                if(index!=0){
                                    if(CDF_ion[index]-random2 <= random2 - CDF_ion[index-1]){x0=index;}
                                    if(CDF_ion[index]-random2 > random2 -  CDF_ion[index-1]){x0=index-1;}
                                }
                                else {x0=0;}
                                if(x0<0){x0=0;}
                                if(x0>=L){x0=L-1;}

              double counter1=0;
              double counter2=0;

    //          qDebug()<< "random walk lacune";
              //random walk lacune
              int timecount_p=0;        //QUI CCE_FRONTAL_ION E CONTATORI DA RIDEFINIRE!
              int timecount_n=0;
              int time_arr_p=0;
              int time_arr_n=0;
              x=x0;
              int j1=0;
              int j2=0;

      //        qDebug()<< "check time_arr_p";
              while(j1==0)
                   {
                   if(x<=0)
                   {
                       while(time_arr_p<num_t_plot)
                       {
                       cce_frontal_ion_p[time_arr_p]+= -(gunn[0]-gunn[x0]);
                       time_arr_p++;
                       }
                       break;

                   }

                   if(x>=L)
                   {
                       while(time_arr_p<num_t_plot)
                       {
                       cce_frontal_ion_p[time_arr_p]+= -(gunn[L-1]-gunn[x0]);
                       time_arr_p++;
                       }
                       break;
                   }
 //                   qDebug()<< "check taup";
                   if(genrand_real1()<ptaup[x])
                   {
                       while(time_arr_p<num_t_plot)
                       {
                       cce_frontal_ion_p[time_arr_p]+= -(gunn[x]-gunn[x0]);
                       time_arr_p++;
                       }
                       break;
                   }

                   if(genrand_real1() > pdrift[x]){x--;}  else {x++;}

//qDebug()<< "check cce_time_frontal_p vs timecount_p";
                   if(timecount_p==cce_time_frontal_p[time_arr_p])
                   {

                       if(x<L && x>=0){cce_frontal_ion_p[time_arr_p]+= -(gunn[x]-gunn[x0]);}
                       else if(x>=L){cce_frontal_ion_p[time_arr_p]+= -(gunn[L-1]-gunn[x0]);}
                       else if(x<0){cce_frontal_ion_p[time_arr_p]+= -(gunn[0]-gunn[x0]);}


                   time_arr_p++;
                   if(time_arr_p==num_t_plot) {break;}
                   }
timecount_p++;

              }
              if(x<L && x>=0){counter1= -(gunn[x]-gunn[x0]);}
              else if(x>=L){counter1= -(gunn[L-1]-gunn[x0]);}
              else if(x<0){counter1= -(gunn[0]-gunn[x0]);}

                   counterp+=counter1;

 //                  qDebug()<< "randomwalk elettroni";
               //random walk elettroni

               x=x0;

               while(j2==0)
                   {
                   if(x>=L)
                   {
                      // x=L-1;
//qDebug()<< "check cce_frontal_p ";
                       while(time_arr_n<num_t_plot)
                       {
                            qDebug()<< time_arr_n;
                       cce_frontal_ion_n[time_arr_n]+= (gunn[L-1]-gunn[x0]);
                       time_arr_n++;

                       }
//qDebug()<< "uscito dal while per cce_frontal_ion_n";
                       break;
                   }
                   else if(x<=0)
                   {

                       while(time_arr_n<num_t_plot)
                       {
                       cce_frontal_ion_n[time_arr_n]+= (gunn[0]-gunn[x0]);
                       time_arr_n++;
                       }

                       break;
                   }

 //                  qDebug()<< "check taun";
                   if(genrand_real1()<ptaun[x])
                   {

                       while(time_arr_n<num_t_plot)
                       {
                       cce_frontal_ion_n[time_arr_n]+= (gunn[x]-gunn[x0]);
                       time_arr_n++;
                       }
                       break;
                   }

                   if(genrand_real1() > ndrift[x]){x--;}  else {x++;}
//qDebug()<< "check cce_time_frontal_n vs timecount_n";

                   if(timecount_n==cce_time_frontal_n[time_arr_n])
                   {
                       if(x<L && x>=0){
                   cce_frontal_ion_n[time_arr_n]+= (gunn[x]-gunn[x0]);}
                       else if(x>=L){cce_frontal_ion_n[time_arr_n]+= (gunn[L-1]-gunn[x0]);}
                       else if(x<0){cce_frontal_ion_n[time_arr_n]+= (gunn[0]-gunn[x0]);}
                   time_arr_n++;
                   if(time_arr_n==num_t_plot){break;}

                   }
                    timecount_n++;

                   }
               if(x<L && x>=0){
            counter2= (gunn[x]-gunn[x0]);}
               else if(x>=L){ counter2= (gunn[L-1]-gunn[x0]);}
               else if(x<0){ counter2= (gunn[0]-gunn[x0]);}

                   countern+=counter2;
                   ctot+=counter1+counter2;
            }


    //      qDebug()<< "fill cce_tfrontal";

          for(int i=0; i<num_t_plot; i++)
          {
              cce_tfrontal_aux[i]+=(cce_frontal_ion_p[i]+cce_frontal_ion_n[i])/n_eh;
              cce_tfrontal[i]=cce_tfrontal_aux[i]/(ion+1);

          }


  //              qDebug()<< "cce_frontal_append";

          cce_frontal.append(ctot/n_eh);


  //          qDebug()<< "evaluate noise";

         //NOISE
         double aux_a,aux_b,r2;
         do
         {
             aux_a = -1 + 2 * genrand_real1();
             aux_b = -1 + 2 * genrand_real1();
             // see if it is in the unit circle
             r2 = aux_a * aux_a + aux_b * aux_b;
         }
         while (r2 > 1.0 || r2 == 0);
         double noise = e_noise/2.355 * aux_b * sqrt (-2.0 * log (r2) / r2);
         double ccenoise=ctot/n_eh+noise;

 // qDebug()<< "evaluate ccenoise";
//  qDebug()<< ccenoise;

         //THRESHOLD

         if(sqrt(ccenoise*ccenoise)>e_threshold)
         {

             //conteggia nuovo ione
             hit_frontal++;

             aux_mean_frontal+=sqrt(ccenoise*ccenoise);
             cce_mean_frontal=aux_mean_frontal/hit_frontal;
            //conteggia la media
            cce_frontal_thresh.append(sqrt(ccenoise*ccenoise));
            int cce_int=sqrt(ccenoise*ccenoise)*100;

   //              qDebug()<< "prepare histogram";

            if(cce_int<tot_spectrum[0].size()+1)
            {
                tot_spectrum[0][cce_int/*+1*/]++;
            }
            else
            {
                tot_spectrum[0][tot_spectrum[0].size()-1]++;
            }
            if(tot_spectrum[0][cce_int/*+1*/]>maximum[0])
            {
                maximum[0]=tot_spectrum[0][cce_int+1];
            }



x_histo_mean.resize(2);
y_histo_mean.resize(2);
x_histo_mean[0]=cce_mean_frontal;
x_histo_mean[1]=cce_mean_frontal;
y_histo_mean[0]=0;
y_histo_mean[1]=maximum[0];

     qDebug()<< "emit passvalues";
emit passValues();

         }


         progress++;
         ion++;
        }

      else{
         // connect(this, SIGNAL(thread_finished()), this, SLOT(deleteLater()));
          emit thread_finished();
          dataTimer.stop();
          disconnect(&dataTimer, SIGNAL(timeout()), this, SLOT(montecarlo()));

          wrapper->stop_thread();
          thread_status=1;

}

    }
    }
}
}




int x1;

void MyLibraryWrapper::start_thread()
{

    qDebug() << lat_front;
    if(lat_front==0)
    {
    connect(&dataTimer, SIGNAL(timeout()), this, SLOT(montecarlo()));
    }

    if(lat_front==1)
    {
    connect(&dataTimer, SIGNAL(timeout()), this, SLOT(frontal()));
    }


    dataTimer.start(0);
    thread_status=0;
}


void MyLibraryWrapper::stop_thread()
{

    dataTimer.stop();
    if(lat_front==0)
{
    disconnect(&dataTimer, SIGNAL(timeout()), this, SLOT(montecarlo()));
  //  connect(wrapper, SIGNAL(delete_thread()), wrapper,SLOT(deleteLater()) );
  //  emit delete_thread();
  //  connect(this, SIGNAL(thread_finished()), this, SLOT(enableButtons()));
}

    if(lat_front==1)
    {
        disconnect(&dataTimer, SIGNAL(timeout()), this, SLOT(frontal()));
   //     connect(wrapper,SIGNAL(delete_thread()), wrapper,SLOT(deleteLater()) );
   //     emit delete_thread();
   //     connect(this, SIGNAL(thread_finished()), this, SLOT(deleteLater()));
    }
    thread_status=1;
    //emit thread_stopped();

}

void MyLibraryWrapper::montecarlo()
{

    if(thread_status==0)
    {

    if(n_point>0){

  // calculate two new data points:
#if QT_VERSION < QT_VERSION_CHECK(4, 7, 0)
     double key = 0;
#else
  double key = QDateTime::currentDateTime().toMSecsSinceEpoch()/1000.0;
#endif


    prog_show = progress/end_sim*100;
    emit updateProgressbar(prog_show);

  static double lastPointKey = 0;
  if (key-lastPointKey >0.05) // at most add point every 10 ms
  {

      if(ion<n_ion)
        {
      if(p<n_point)
        {

          qDebug() << "p value: " << p;
          int xinit = p*steps;

          //BEAM DISPERSION
          double aux_a1,aux_b1,rstraggl;
          do
             {
             aux_a1 = -1 + 2 * genrand_real1();
             aux_b1 = -1 + 2 * genrand_real1();
             // see if it is in the unit circle
             rstraggl = aux_a1 * aux_a1 + aux_b1 * aux_b1;
             }
             while (rstraggl > 1.0 || rstraggl == 0);
             // Box-Muller transform
             x0 = xinit + (int) floor(dispersion/deltax/2.355 * aux_a1 * sqrt (-2.0 * log (rstraggl) / rstraggl));

          double ctot=0;
          double counterp=0;
          double countern=0;
          int x;

           qDebug() << "azzero cicli cce_frontal_ion";

          for(int j=0; j<n_point; j++)
          {
          for(int i=0; i<num_t_plot; i++)
          {
              cce_frontal_ion_p2[j][i]=0;
              cce_frontal_ion_n2[j][i]=0;
          }
          }

           qDebug() << "simulo portatori";
          for(int h=0;h<n_eh;h++)
            {

              x1=x0;
              //inizializzazione contatori ausiliari per il ciclo interno

              double counter1=0;
              double counter2=0;

              //random walk lacune
              int timecount_p=0;
              int timecount_n=0;
              int time_arr_p=0;
              int time_arr_n=0;
              x=x1;
              int j1=0;
              int j2=0;
              while(j1==0)
                   {
                   if(x<=0){
                       while(time_arr_p<num_t_plot)
                       {
                       cce_frontal_ion_p2[p][time_arr_p]+= -(gunn[x]-gunn[x1]); //Cce_frontal_ion va ridefinito per il lateral
                       time_arr_p++;
                       }
                       break;
                   }

                   if(x>=L){
                       while(time_arr_p<num_t_plot)
                       {
                       cce_frontal_ion_p2[p][time_arr_p]+= -(gunn[x]-gunn[x1]);
                       time_arr_p++;
                       }
                       break;
                   }
                   if(genrand_real1()<ptaup[x]){
                       while(time_arr_p<num_t_plot)
                       {
                       cce_frontal_ion_p2[p][time_arr_p]+= -(gunn[x]-gunn[x1]);
                       time_arr_p++;
                       }
                       break;
                   }

                   if(genrand_real1() > pdrift[x]){x--;}  else {x++;}

                   if(timecount_p==cce_time_frontal_p2[p][time_arr_p])
                   {
                   cce_frontal_ion_p2[p][time_arr_p]+= -(gunn[x]-gunn[x1]);
                   time_arr_p++;
                   if(time_arr_p==num_t_plot) {break;}
                   }
timecount_p++;
      }

               counter1= -(gunn[x]-gunn[x1]);
                   counterp+=counter1;
                //



                   //random walk elettroni
               x=x1;
               while(j2==0)
                   {
                   if(x>=L){
                       while(time_arr_n<num_t_plot)
                       {
                       cce_frontal_ion_n2[p][time_arr_n]+= (gunn[x]-gunn[x1]); //Cce_frontal_ion va ridefinito per il lateral
                       time_arr_n++;
                       }
                       break;}
                   if(x<=0){
                       while(time_arr_n<num_t_plot)
                       {
                       cce_frontal_ion_n2[p][time_arr_n]+= (gunn[x]-gunn[x1]); //Cce_frontal_ion va ridefinito per il lateral
                       time_arr_n++;
                       }
                       break;}
                   if(genrand_real1()<ptaun[x]){
                       while(time_arr_n<num_t_plot)
                       {
                       cce_frontal_ion_n2[p][time_arr_n]+= (gunn[x]-gunn[x1]); //Cce_frontal_ion va ridefinito per il lateral
                       time_arr_n++;
                       }
                       break;}

                   if(genrand_real1() > ndrift[x]){x--;}  else {x++;}

                   if(timecount_n==cce_time_frontal_n2[p][time_arr_n])
                   {
                   cce_frontal_ion_n2[p][time_arr_n]+= (gunn[x]-gunn[x1]);
                   time_arr_n++;
                   if(time_arr_n==num_t_plot) {break;}
                   }
timecount_n++;

                   }

                   counter2= (gunn[x]-gunn[x1]);
                   countern+=counter2;
                   ctot+=counter1+counter2;

            }


         qDebug() << "aggiorno cce_tfrontal2";

          for(int i=0; i<num_t_plot; i++)
          {
              cce_tfrontal_aux2[p][i]+=(cce_frontal_ion_p2[p][i]+cce_frontal_ion_n2[p][i])/n_eh;
              cce_tfrontal2[p][i]=cce_tfrontal_aux2[p][i]/(ion+1);

          }

         cce.append(ctot/n_eh);


          qDebug() << "aggiorno noise";
         //NOISE
         double aux_a,aux_b,r2;
         do
         {
             aux_a = -1 + 2 * genrand_real1();
             aux_b = -1 + 2 * genrand_real1();
             // see if it is in the unit circle
             r2 = aux_a * aux_a + aux_b * aux_b;
         }
         while (r2 > 1.0 || r2 == 0);
         double noise = e_noise/2.355 * aux_b * sqrt (-2.0 * log (r2) / r2);
         double ccenoise=ctot/n_eh+noise;
         cce_noise.append(ccenoise);

         //THRESHOLD

          qDebug() << "aggiorno threshold";
         if(sqrt(ccenoise*ccenoise)>e_threshold)
         {

             //conteggia nuovo ione
            hit_over_threshold[p]++;
            cce_mean[p]+=sqrt(ccenoise*ccenoise);

            //conteggia la media
            if(hit_over_threshold[p]>0)
            {
            cce_mean_plot[p]=cce_mean[p]/hit_over_threshold[p];

            }
            else
            {
                cce_mean_plot[p]=0;
            }

             tot_hit_over=0;
             tot_mean_plot=0;

            for(int q=0; q<n_point; q++)
            {
                tot_hit_over+=hit_over_threshold[q];
                tot_mean_plot+=cce_mean[q];
                qDebug() << tot_hit_over << "    " << tot_mean_plot << "   " << tot_mean_plot/tot_hit_over;
            }

            cce_threshold[p].append(ccenoise);
            multipos[p].append(xinit*deltax*1e6);
            int cce_int=sqrt(ccenoise*ccenoise)*100;

             qDebug() << "aggiorno spettro";

            if(cce_int<tot_spectrum[p+1].size()+1){tot_spectrum[p+1][cce_int]++;} else{tot_spectrum[p+1][tot_spectrum.size()-1]++;}
            if(tot_spectrum[p+1][cce_int]>maximum[p+1]) {maximum[p+1]=tot_spectrum[p+1][cce_int];}

            if(cce_int<tot_spectrum[0].size()+1){tot_spectrum[0][cce_int]++;} else{tot_spectrum[0][tot_spectrum.size()-1]++;}
            if(tot_spectrum[0][cce_int]>maximum[0]) {maximum[0]=tot_spectrum[0][cce_int];}


            //registra nuovo ione su cce_thresh, con posizione su multipos[p] che serve solo per il plot
            //sort di cce_threshold per mediana: DUPLICA CCHE_THRESHOLD per evitare di sovrascriverlo

            double a,b;



             for(int j=0; j< n_point; j++)
             {
                for(int i=0; i< cce_threshold[j].size()-1; i++)
                {



                  if(cce_threshold[j][i]>cce_threshold[j][i+1])
                  {
                    a=cce_threshold[j][i];
                    b=cce_threshold[j][i+1];
                    cce_threshold[j][i]=b;
                    cce_threshold[j][i+1]=a;
                  }


                }
              }

              qDebug() << "calcolo mediana";

             //ora calcola la mediana di cce_threshold

             int l1;

             for(int j=0; j< n_point; j++)
               {

                l1 = cce_threshold[j].size();

                if(l1 == 0) cce_median[j]=0;
                else{
                   if(l1 % 2 == 1)
                   {cce_median[j]=cce_threshold[j][l1/2];}
                   else {cce_median[j]=(cce_threshold[j][l1/2-1]+cce_threshold[j][l1/2])/2;}
}

              }

         }



         progress++;
          p++;

           qDebug() << "incrementato p";

          emit passValues();

         qDebug() << "emesso passvalues";

        }
      else{p=0; ion++;}


        }
      else{
//      connect(this, SIGNAL(
          //finished()), this, SLOT(deleteLater()));
    //  emit thread_finished();



}

      lastPointKey = key;

    }
    }
}

}


void MainWindow::on_pushButton_stop_clicked()
{


    disconnect(wrapper, SIGNAL(updateProgressbar(int)),ui->progressBar, SLOT(setValue(int)));
  //  disconnect(wrapper, SIGNAL(thread_finished()),this,SLOT(wrapperDone()));
    wrapper->stop_thread();

    ui->pushButton_resume->setEnabled(true);
    ui->pushButton_sim->setEnabled(true);
    ui->pushButton_stop->setDisabled(true);
    ui->pushButton_simsetup->setEnabled(true);

    ui->textMessageBox->append("Simulation stopped.");


    }

void MainWindow::on_pushButton_saveplot_clicked()
{

    FileBrowser browser;
    browser.exec();
    QString dir=browser.on_buttonBox_accepted();
    dir.append("/export");
    dir.append(QDateTime::currentDateTime().toString());
    QDir directory;
    directory.mkpath(dir);

    QString path_readme = dir;
    path_readme.append("/simulation_contents.txt");
    QFile file_readme(path_readme);
    file_readme.open(QIODevice::WriteOnly | QIODevice::Text);
    QTextStream out_readme(&file_readme);

    out_readme << "Device Setup\n\n";
    out_readme << "Path to electric field profile: " << path_efield << " \n";
    out_readme << "Path to electric weighting potential profile: " << path_gunn << " \n";
    out_readme << "deltax: " << deltax << " m\n";
    out_readme << "deltatn: " << deltatn << " s\n";
    out_readme << "deltatp: " << deltatp << " s\n";
    out_readme << "electron mobility: " << mun << " m2/V/s\n";
    out_readme << "hole mobility: " << mup << " m2/V/s\n";
    out_readme << "electron diffusivity: " << ndiff << " m2/s\n";
    out_readme << "hole diffusivity: " << pdiff << " m2/s\n\n\n";
    out_readme << "Simulation setup\n";
    if(lat_front==1)
    {
        out_readme << "Frontal IBIC geometry.\n";
        out_readme << "Path to ionization profile: " << path_ioniz << " \n";
    }
    else
    {
        out_readme << "Lateral IBIC geometry.\n";
    }
    out_readme << "number of generation points: " << n_point << "\n";
    out_readme << "number of ions per point: " << n_ion << "\n";
    out_readme << "number of electron-hole paris per ion: " << n_eh << "\n";
    out_readme << "beam spot size: " << dispersion << "um\n";
    out_readme << "electronic noise: " << e_noise*100 << "%\n";
    out_readme << "electronic threshold: " << e_threshold*100 << "%\n\n";
     out_readme << "Irradiation setup\n";

     if(radiation_control==1)
     {
         out_readme << "Radiation damage considered .\n";
        out_readme << "Path to vacancy density profile: " << path_vac << " \n";
        out_readme << "ion fluence: " << fluence*1e-4 << " cm2\n";
        out_readme << "electron trap-to-vacancy ratio: " << radhard_n << "\n";
        out_readme << "hole trap-to-vacancy ratio: " << radhard_p << "\n";
        out_readme << "electron thermal velocity: " << ntherm*100 << "\n";
        out_readme << "hole thermal velocity: " << ptherm*100 << " cm/s\n";
        out_readme << "electron trapping cross section: " << nsigma*1e4 << " cm2\n";
        out_readme << "hole trapping cross section: " << psigma*1e4 << " cm2\n";

     }
     else
     {
         out_readme << "Radiation damage not considered.\n";
     }

    file_readme.close();

    QString path_threshold = dir;
    path_threshold.append("/cce_pulses.txt");

    QFile file_thresh(path_threshold);
    file_thresh.open(QIODevice::WriteOnly | QIODevice::Text);
    QTextStream out_thresh(&file_thresh);

    if(lat_front==0)
    {
    for(int k=0; k<cce_threshold.size(); k++)
    {
        for(int j=0; j<cce_threshold[k].size(); j++)
        {
            out_thresh << multipos[k][j] << "\t" <<  cce_threshold[k][j]  << "\n";
        }
    }

    }
    else
    {

        for(int j=0; j<cce_frontal_thresh.size(); j++)
        {
            out_thresh << cce_frontal_thresh[j]  << "\n";
        }
    }
    file_thresh.close();

    QString path_spectrum = dir;
    path_spectrum.append("/spectrum.txt");
    QFile file_spectrum(path_spectrum);

    file_spectrum.open(QIODevice::WriteOnly | QIODevice::Text);
    QTextStream out_spectrum(&file_spectrum);

    if(gnearest<0)
    {

        for(int i=0; i<tot_spectrum[0].size(); i++)
        {
            out_spectrum << histo_step[i] << "\t" <<  tot_spectrum[0][i]  << "\n";
        }

    }
    else
    {
    for(int i=0; i<tot_spectrum[gnearest].size(); i++)
    {
        out_spectrum << histo_step[i] << "\t" <<  tot_spectrum[gnearest+1][i]  << "\n";
    }

    }

    file_spectrum.close();

    if(lat_front==0)
    {
    QString path_ccemean = dir;
    path_ccemean.append("/cce_mean_profile.txt");
    QFile file_ccemean(path_ccemean);
    QString path_cce_median = dir;
    path_cce_median.append("/cce_median_profile.txt");
    QFile file_ccemedian(path_cce_median);

    file_ccemean.open(QIODevice::WriteOnly | QIODevice::Text);
    QTextStream out_ccemean(&file_ccemean);
    file_ccemedian.open(QIODevice::WriteOnly | QIODevice::Text);
    QTextStream out_ccemedian(&file_ccemedian);


    for(int i=0; i<n_point; i++)
    {
        out_ccemean << i*steps*deltax << "\t" <<  cce_mean_plot[i]  << "\n";
        out_ccemedian << i*steps*deltax << "\t" <<  cce_median[i]  << "\n";
    }

    file_ccemean.close();
    file_ccemedian.close();
    }

    if(radiation_control==1)
    {
    QString path_vac = dir;
    path_vac.append("/vac.txt");
    QFile file_vac(path_vac);
    file_vac.open(QIODevice::WriteOnly | QIODevice::Text);
    QTextStream out_vac(&file_vac);
    for(int k=0; k<L; k++)
    {
        out_vac << k*deltax << "\t" <<    yioniz_inter[k]  << "\n";
    }


    file_vac.close();
    }


    if(lat_front==1)
    {
    QString path_ioniz = dir;
    path_ioniz.append("/ioniz.txt");
    QFile file_ioniz(path_ioniz);
    file_ioniz.open(QIODevice::WriteOnly | QIODevice::Text);
    QTextStream out_ioniz(&file_ioniz);
    for(int k=0; k<L; k++)
    {
        out_ioniz << k*deltax << "\t" <<  yioniz_inter[k]  << "\n";
    }

    file_ioniz.close();
    }






    QString path_taup = dir;
    path_taup.append("/taup.txt");
    QFile file_taup(path_taup);
    file_taup.open(QIODevice::WriteOnly | QIODevice::Text);
    QTextStream out_taup(&file_taup);

    QString path_taun = dir;
    path_taun.append("/taun.txt");
    QFile file_taun(path_taun);
    file_taun.open(QIODevice::WriteOnly | QIODevice::Text);
    QTextStream out_taun(&file_taun);

    for(int k=0; k<L; k++)
    {
        out_taup << k*deltax << "\t" <<  taup_arr[k]  << "\n";
        out_taun << k*deltax << "\t" <<  taun_arr[k]  << "\n";

    }

    file_taun.close();
    file_taup.close();

    QString path_ndrift = dir;
    path_ndrift.append("/ndrift.txt");
    QFile file_ndrift(path_ndrift);
    file_ndrift.open(QIODevice::WriteOnly | QIODevice::Text);
    QTextStream out_ndrift(&file_ndrift);

    QString path_pdrift = dir;
    path_pdrift.append("/pdrift.txt");
    QFile file_pdrift(path_pdrift);
    file_pdrift.open(QIODevice::WriteOnly | QIODevice::Text);
    QTextStream out_pdrift(&file_pdrift);

    for(int k=0; k<L; k++)
    {
        out_pdrift << k*deltax << "\t" <<  pdrift[k]  << "\n";
        out_ndrift << k*deltax << "\t" <<  ndrift[k]  << "\n";
    }

    file_ndrift.close();
    file_pdrift.close();

    QString path_outefield = dir;
    path_outefield.append("/efield.txt");
    QFile file_outefield(path_outefield);
    file_outefield.open(QIODevice::WriteOnly | QIODevice::Text);
    QTextStream out_outefield(&file_outefield);

    QString path_outgunn = dir;
    path_outgunn.append("/gunn.txt");
    QFile file_outgunn(path_outgunn);
    file_outgunn.open(QIODevice::WriteOnly | QIODevice::Text);
    QTextStream out_outgunn(&file_outgunn);

    for(int k=0; k<L; k++)
    {
        out_outefield << k*deltax << "\t" <<  efield[k]  << "\n";
        out_outgunn << k*deltax << "\t" <<  gunn[k]  << "\n";

    }

    file_outefield.close();
    file_outgunn.close();



    QString path_risetime = dir;
    path_risetime.append("/time_resolved_CCE.txt");
    QFile file_risetime(path_risetime);
    file_risetime.open(QIODevice::WriteOnly | QIODevice::Text);
    QTextStream out_risetime(&file_risetime);



    for(int k=0; k<num_t_plot; k++)
    {
        if(lat_front==1)
        {
        out_risetime << cce_time_frontal_tot[k] << "\t" <<  cce_tfrontal[k]  << "\n";
        }

        if(lat_front==0)
        {

        if(gnearest<0)
        {
            out_risetime << cce_time_frontal_tot[k] << "\t" <<  cce_tfrontal2[0][k]  << "\n";
        }
        else
        {
        out_risetime << cce_time_frontal_tot[k] << "\t" <<  cce_tfrontal2[gnearest+1][k]  << "\n";
        }
}
    }

    file_risetime.close();

}



void MainWindow::electrostatics(){


    gunn.resize(0);
    pos_gunn.resize(0);

    QFile file_gunn(path_gunn);
    QTextStream its_gunn(&file_gunn);
    QString line_gunn;

    if (!file_gunn.open(QIODevice::ReadOnly))
    {
     qDebug() << "file not found" << endl;
    }


int i=0;
double space_step_gunn=0;
int array_step_count=0;

    while( !its_gunn.atEnd() )
    {

        line_gunn = its_gunn.readLine();
        line_gunn.replace(",",".");
        line_gunn.replace("\t", "    ");
        QStringList list_gunn=line_gunn.split(" ",QString::SkipEmptyParts);

        if(list_gunn.size()>1)
        {
        bool validate1, validate2;

        list_gunn.first().toDouble(&validate1);
        list_gunn.last().toDouble(&validate2);

        if (validate1 == true && validate2 ==true && list_gunn.size()>1)
        {
        pos_gunn.push_back(list_gunn.first().toDouble()); // m
        gunn.push_back(list_gunn.last().toDouble()); //m

        if(i==1){space_step_gunn=(pos_gunn[i]-pos_gunn[i-1]);}
        if(i>0)
        {if(fabs(pos_gunn[i]-pos_gunn[i-1] -space_step_gunn) >1e-9 ) {array_step_count++;}}

        i++;

        }
        }
    }

      int control_deltax_gunn=-1;
      int control_deltax_efield=-1;
      int control_deltax=-1;

    if (array_step_count==0)
      {
        deltax=space_step_gunn;
        control_deltax_gunn=0;
      }



    file_gunn.close();


    efield.resize(0);
    pos_efield.resize(0);

    QFile file_efield(path_efield);
    QTextStream its_efield(&file_efield);
    QString line_efield;

    if (!file_efield.open(QIODevice::ReadOnly))
    {
     qDebug() << "file not found" << endl;
    }

    i=0;
    double space_step_efield=0;
    array_step_count=0;

    while( !its_efield.atEnd() )
    {

        line_efield = its_efield.readLine();
        line_efield.replace(",",".");
        line_efield.replace("\t", "    ");
        QStringList list_efield=line_efield.split(" ",QString::SkipEmptyParts);


        if(list_efield.size()>1)
        {


        bool validate1, validate2;

        list_efield.first().toDouble(&validate1);
        list_efield.last().toDouble(&validate2);

        if (validate1 == true && validate2 ==true && list_efield.size()>1)
        {


        pos_efield.push_back(list_efield.first().toDouble()); //espresso in m
        efield.push_back(list_efield.last().toDouble()*100); //convertito in V/m da input in V/cm

        if(i==1){space_step_efield=pos_efield[i]-pos_efield[i-1];}
        if(i>0)
        {if(fabs(pos_efield[i]-pos_efield[i-1] -space_step_efield) >1e-9 )  {array_step_count++;}}


        i++;
        }
        }

    }


    if (array_step_count==0)
      {
        control_deltax_efield=0;
      }

    if(space_step_gunn==space_step_efield)
      {control_deltax=0;}

    file_efield.close();


    int control_array=-1;
    int control_tau=-1;
    int control_mob=-1;
    int control_vsat=-1;
    int control_temp=-1;

    int control=-1;


    //CHECK SU DELTAX!

    if(efield.size()==gunn.size() && gunn.size() > 0) {
        control_array=0;
        L=gunn.size();
        ui->textMessageBox->append("Input files imported successfully. Device lenght is " + QString::number(L) + "space steps.");
      }

    if(taun > 0 && taup > 0)
      {
        control_tau=0;
        ui->textMessageBox->append("Carriers lifetime defined successfully. taun=" + QString::number(taun) +" s, taup=" + QString::number(taup) + " s.");
      }

    taup_arr.resize(L);
    taun_arr.resize(L);
    ptaup.resize(L);
    ptaun.resize(L);

    if(yvac_inter.size()!=L)
    {
        if(yvac_inter.size()==0)
        {
            yvac_inter.resize(L);
            for(int i=0; i<L; i++)
            {
                yvac_inter[i]=0;
            }

        }
        else
        {
        yvac_inter.resize(L);
        for(int i=0; i<L; i++)
        {
            int j=0;

            while(i*deltax*1e6 > x_vac[j]) {j++;}

            if(j==0)
            {
                yvac_inter[i]=y_vac[j];
            }
            else if((i*deltax*1e6-x_vac[j-1])<(x_vac[j]-i*deltax*1e6))
            {
                yvac_inter[i]=y_vac[j-1];
            }
            else
            {
                yvac_inter[i]=y_vac[j];
            }

        }

        }




    }

    tauplot_scale_x=L*deltax*1e6;
    ui->tau_x->setMaximum(L*deltax*1e6);
ui->tau_x->setValue(tauplot_scale_x);
ui->tau_x->setSingleStep(1);

tauplot_scale_y=1.5*max(taup,taun)*1e9;

ui->tau_y->setMaximum(tauplot_scale_y);
ui->tau_y->setValue(tauplot_scale_y);
ui->tau_y->setSingleStep(1);


    if(mup > 0 && mun > 0)
      {
        control_mob=0;
        ui->textMessageBox->append("Carriers mobility defined successfully. Dn=" + QString::number(mun) + " m2/V/s, Dp=" +QString::number(mup)+ " m2/V/s.");
      }

    if(vsatn > 0 && vsatp > 0)
      {
        control_vsat=0;
        ui->textMessageBox->append("Saturation velocities  defined successfully. vsatn=" + QString::number(vsatn*100) + " cm/s, vsatp=" +QString::number(vsatp*100)+ " cm/s.");
      }

    if(temp > 0)
      {
        control_temp=0;
        ui->textMessageBox->append("Temperature defined successfully. T=" + QString::number(temp) + " K.");
      }


    if(control_deltax == 0)
      {
        ui->textMessageBox->append("Space step size evaluated successfully. deltax=" + QString::number(deltax) + " m.");
      }


    // QUI INSERISCI CONTROLLO LOGICA

    if(control_array==0 && control_tau==0 && control_vsat==0 && control_temp==0 && control_deltax ==0 && control_deltax_gunn ==0 && control_deltax_efield ==0
) {

        control=0;
        ui->textMessageBox->append("Parameters initialized successfully!");
        ui->pushButton_simsetup->setEnabled(true);
      }
    else{
        if(control_array<0) {ui->textMessageBox->append("ERROR: Array lenght is zero, or arrays don't have the same lenght.");}
        if(control_tau<0){ui->textMessageBox->append("ERROR: Carrier lifetimes are undefined.");}
        if(control_vsat<0){ui->textMessageBox->append("ERROR: Saturation velocities are undefined.");}
        if(control_temp<0){ui->textMessageBox->append("ERROR: Temperature is undefined.");}
        if(control_deltax_gunn<0){ui->textMessageBox->append("ERROR: Irregular space size in weighting potential input file.");}
        if(control_deltax_efield<0){ui->textMessageBox->append("ERROR: Irregular space size in electric field input file.");}
        if(control_deltax<0){ui->textMessageBox->append("ERROR: Space step size for electric field and weighting potential don't match.");}
        ui->textMessageBox->append("Simulation cannot be started.");
      }
    //manca lunghezza in unità di misura standard


    double k_q=8.62e-5; // in K^(-1), se kb/q=8.62e-5 eV/K
    ndiff=k_q*mun*temp; // V/K*K*m2/v/s
    pdiff=k_q*mup*temp;

    ui->textMessageBox->append("Diffusivity evaluated successfully. Dn=" + QString::number(ndiff) + " m2/s, Dp=" + QString::number(pdiff) + " m2/s.");

    pdrift.resize(gunn.size());
    ndrift.resize(gunn.size());

    deltatn=deltax*deltax/2/ndiff;  // m2/m2*s massima consenstita! (warning)
    deltatp=deltax*deltax/2/pdiff;

    ui->textMessageBox->append("Time step sizes evaluated successfully. deltatn=" + QString::number(deltatn) + " s, deltatp=" + QString::number(deltatp) + " s.");
    ui->textMessageBox->append("Maximum holding time evaluated successfully.");


mn.resize(L);
mp.resize(L);

    for(int i=0; i<gunn.size(); i++){

       mn[i]=mun/(sqrt(1+mun*mun*efield[i]*efield[i]/vsatn/vsatn));
       mp[i]=mup/(sqrt(1+mup*mup*efield[i]*efield[i]/vsatp/vsatp));

        //includes formula for saturation velocity //DA RIVEDERE!!!

        pdrift[i]=0.5+efield[i]*mp[i]/2*deltatp/deltax;
        ndrift[i]=0.5-efield[i]*mn[i]/2*deltatn/deltax;
      }

setupGunn(ui->gunn_plot);
ui->gunn_plot->replot();

setupEfield(ui->efield_plot);
ui->efield_plot->replot();

    setupNdrift(ui->ndrift_plot);
    ui->ndrift_plot->replot();

    setupPdrift(ui->pdrift_plot);
    ui->pdrift_plot->replot();


ui->pushButton_sim->setDisabled(true);

ui->time_spinBox->setDisabled(true);

text_transport();



}






void MainWindow::on_checkBox_clicked()
{


   if(ui->checkBox->isChecked()==false) {set_legend=0; ui->mc_plot->legend->setVisible(false);}
     if(ui->checkBox->isChecked()==true) {set_legend=1; ui->mc_plot->legend->setVisible(true);}
    ui->mc_plot->replot();

}



void MainWindow::on_horizontalSlider_valueChanged(int value)
{
    //xgpoint.resize(2);
    slide_generation_point=value;
    int nearest;

  if(value!=-1)
    {
     nearest=slide_generation_point/steps;
    ui->horizontalSlider->setValue(nearest*(steps));
    }

else {nearest=-100;}
    gnearest=nearest;

    gpoint=(double) (nearest)*(steps*(n_point)-1)*deltax/n_point*1e6;
   if(this->ui->horizontalSlider->value()==-1){ this->ui->textBrowser->setText("All data");}
    else{this->ui->textBrowser->setText("x=" + QString::number(gpoint)+" um");}
    xgpoint[0]=gpoint;
    xgpoint[1]=gpoint;

    ui->mc_plot->graph(3)->clearData();
    ui->mc_plot->graph(5)->clearData();
    ui->mc_plot->graph(5)->setData(xgpoint,ygpoint);
    ui->mc_plot->replot();
    ui->time_plot->replot();
    ui->histo_plot->clearItems();
    ui->histo_plot->graph(0)->clearData();
    ui->histo_plot->graph(1)->clearData();


    QVector<QVector<double> > x_histo_mean_lateral, y_histo_mean_lateral;
    x_histo_mean_lateral.resize(n_point+1);
    y_histo_mean_lateral.resize(n_point+1);

    x_histo_mean_lateral[0].resize(2);
    y_histo_mean_lateral[0].resize(2);
    if(tot_hit_over>0){
    x_histo_mean_lateral[0][0]=tot_mean_plot/tot_hit_over;
    x_histo_mean_lateral[0][1]=tot_mean_plot/tot_hit_over;
    }
    else
    {
        x_histo_mean_lateral[0][0]=0;
        x_histo_mean_lateral[0][1]=0;
    }
    y_histo_mean_lateral[0][0]=0;
    y_histo_mean_lateral[0][1]=maximum[0];

    for(int q=1; q<n_point; q++)
    {
        x_histo_mean_lateral[q].resize(2);
        y_histo_mean_lateral[q].resize(2);
        x_histo_mean_lateral[q][0]=cce_mean_plot[q-1];
        x_histo_mean_lateral[q][1]=cce_mean_plot[q-1];
        y_histo_mean_lateral[q][0]=0;
        y_histo_mean_lateral[q][1]=maximum[q];

    }


    if(gnearest<0){
    ui->histo_plot->graph(0)->setData(histo_step, tot_spectrum[0]);
    ui->histo_plot->graph(1)->setData(x_histo_mean_lateral[0], y_histo_mean_lateral[0]);
    ui->histo_plot->yAxis->setRange(-0.0,maximum[0]*1.2);

    QCPItemText *textLabel = new QCPItemText(ui->histo_plot);
    ui->histo_plot->addItem(textLabel);
    textLabel->position->setType(QCPItemPosition::ptAxisRectRatio);
    textLabel->position->setCoords(0.5,0);
    textLabel->setText("Mean: " + QString::number(tot_mean_plot/tot_hit_over));
    textLabel->setPen(QPen(Qt::black));
    textLabel->setPositionAlignment(Qt::AlignTop|Qt::AlignHCenter);
    }
    else
    {
    ui->histo_plot->graph(0)->setData(histo_step, tot_spectrum[gnearest+1]);
    ui->histo_plot->graph(1)->setData(x_histo_mean_lateral[gnearest+1], y_histo_mean_lateral[gnearest+1]);
    ui->histo_plot->yAxis->setRange(-0.0,maximum[gnearest+1]*1.2);

    ui->histo_plot->yAxis->setRange(-0.0,maximum[gnearest+1]+10);
    QCPItemText *textLabel = new QCPItemText(ui->histo_plot);
    ui->histo_plot->addItem(textLabel);
    textLabel->position->setType(QCPItemPosition::ptAxisRectRatio);
    textLabel->position->setCoords(0.5,0);
    textLabel->setText("Mean: " + QString::number(cce_mean_plot[gnearest]));
    textLabel->setPen(QPen(Qt::black));
    textLabel->setPositionAlignment(Qt::AlignTop|Qt::AlignHCenter);
    }



        ui->histo_plot->replot();


        ui->time_plot->graph(0)->clearData();
        if(gnearest>=0)
        {
        ui->time_plot->graph(0)->setData(cce_time_frontal_tot,cce_tfrontal2[gnearest]);
        ui->time_plot->xAxis->setRange(0,valtscale);
        ui->time_plot->yAxis->setRange(0,1.1);
}
ui->time_plot->replot();


}

void MainWindow::on_time_spinBox_editingFinished()
{

valtscale=ui->time_spinBox->value();
ui->time_plot->xAxis->setRange(0,valtscale);
ui->time_plot->replot();

}

void MainWindow::on_time_spinBox_valueChanged(/*int arg1*/)
{
    valtscale=ui->time_spinBox->value();
    ui->time_plot->xAxis->setRange(0,valtscale);
    ui->time_plot->replot();
}


void MainWindow::text_transport(){

    ui->textBrowser_transport->clear();
    ui->textBrowser_transport->setText("deltax: " + QString::number(deltax) + " m");
    ui->textBrowser_transport->append("deltatn: " + QString::number(deltatn)+ " s");
    ui->textBrowser_transport->append("deltatp: " + QString::number(deltatp) + " s");
    ui->textBrowser_transport->append("mun: " + QString::number(mun)+ " m2/V/s");
    ui->textBrowser_transport->append("mup: " + QString::number(mup)+ " m2/V/s");
    ui->textBrowser_transport->append("Dn: " + QString::number(ndiff)+ " m2/s");
    ui->textBrowser_transport->append("Dp: " + QString::number(pdiff)+ " m2/s");
}

void MainWindow::text_simulation(){

    ui->textBrowser_simulation->clear();
    if(lat_front==1)
    {
    ui->textBrowser_simulation->setText("Frontal IBIC simulation");
    }
    if(lat_front==0)
    {
    ui->textBrowser_simulation->setText("Lateral IBIC simulation");
    }

    ui->textBrowser_simulation->append("points: " + QString::number(n_point));
    ui->textBrowser_simulation->append("ions per point: " + QString::number(n_ion));
    ui->textBrowser_simulation->append("pairs per ion: " + QString::number(n_eh) );
    ui->textBrowser_simulation->append("beam size: " + QString::number(dispersion)+ " um");
    ui->textBrowser_simulation->append("noise: " + QString::number(e_noise*100)+ " %");
    ui->textBrowser_simulation->append("threshold: " + QString::number(e_threshold*100)+ " %");
}

void MainWindow::text_radhard(){

    ui->textBrowser_irradiation->clear();
    if(radiation_control==0)
    {
        ui->textBrowser_irradiation->setText("No irradiation considered.");
    }

    if(radiation_control==1)
    {
    ui->textBrowser_irradiation->setText("ion fluence: " + QString::number(fluence*1e-4)+ " 1/cm2" );
    ui->textBrowser_irradiation->append("n trap/vacancy: " + QString::number(radhard_n));
    ui->textBrowser_irradiation->append("p trap/vacancy: " + QString::number(radhard_p));
    ui->textBrowser_irradiation->append("n therm velocity: " + QString::number(ntherm)+ " m/s");
    ui->textBrowser_irradiation->append("p therm velocity: " + QString::number(ptherm)+ " m/s");
    ui->textBrowser_irradiation->append("n cross section: " + QString::number(nsigma)+ " m2");
    ui->textBrowser_irradiation->append("p cross section: " + QString::number(psigma)+ " m2");
}

}


void MainWindow::on_pushButton_radsetup_clicked()
{
    Radhard_dialog dialog;

    //    dialog.path_vac=path_vac; COME LO FAI?

   // copia ancora, per lanciare all'avvio, i vettori di vacanze così da plottarli al volo.


    dialog.exec();

    if(dialog.control_window==0)
    {

    fluence=dialog.fluence*1e4; //check udm
    radhard_p=dialog.radhard_p;
    radhard_n=dialog.radhard_n;
    nsigma=dialog.sigman*1e-4;
    psigma=dialog.sigmap*1e-4;
    ptherm=dialog.ptherm/100;
    ntherm=dialog.ntherm/100;
    radiation_control=dialog.radiation_control;
    control_vac=dialog.control_window;
    }

    if(radiation_control==1)
    {

    path_vac=dialog.path_vac;


    QFile file_vac(path_vac);
    QTextStream its_vac(&file_vac);
    QString line_vac;

    if (!file_vac.open(QIODevice::ReadOnly))
        {
        ui->textMessageBox->append("Vacancy file not found");
        control_vac=1;
    }

    qDebug()<< "importato parametri da file di dialogo";
    if(control_vac==0)
    {
        x_vac=dialog.x_vac;
        y_vac=dialog.y_vac;

      /*
        int l=0;

        while(!its_vac.atEnd())
        {


        line_vac = its_vac.readLine();
        line_vac.replace(",",".");
        line_vac.replace("\t", "   ");
        QStringList list_vac=line_vac.split(" ",QString::SkipEmptyParts);

        if(list_vac.size()>0)
        {
        bool validate1, validate2;

        list_vac.first().toDouble(&validate1);
        list_vac.last().toDouble(&validate2);

        if (validate1 == true && validate2 ==true && list_vac.size()>1)
        {
        x_vac[l]=list_vac.first().toDouble()*1e-4-dead_layer;  //inserito già in micrometri
        y_vac[l]=list_vac.last().toDouble()*1e10;            //udm: vac/m
        l++;
        }

        }
        }


    file_vac.close();*/
    }
    }

    else
    {
        for(int l=0; l<x_vac.size(); l++)
        {

            x_vac[l]=0;  //inserito già in micrometri
            x_vac[l]=l/*-dead_layer*/;
            y_vac[l]=0;
            control_vac=0;

        }
    }
    qDebug()<<"importato file SRIM";

    yvac_inter.resize(L);

    for(int i=0; i<L; i++)
    {
    yvac_inter[i]=0;
    }

qDebug()<<"parto con l'interpolazione dell'array vacanze";

    int i=0;
    for(i=0; i<L; i++)
    {
        int j=0;

        while(i*deltax*1e6 > x_vac[j] && j< x_vac.size()) {
            j++;
            //qDebug()<<j;
        }

        if(j==0)
        {
            yvac_inter[i]=y_vac[j];
   /*         qDebug() << "marker 2";
            qDebug()<< j;*/
        }
        else if(j==x_ioniz.size()-1)
        {
            yvac_inter[i]=0;
          /*  qDebug()<<"marker3";
            qDebug()<<yvac_inter[i];*/
            break;
        }
        else if((i*deltax*1e6-x_vac[j-1])<(x_vac[j]-i*deltax*1e6) )
        {
            yvac_inter[i]=y_vac[j-1];
        }
        else
        {
            yvac_inter[i]=y_vac[j];
        }     
    }

    qDebug()<< "metto a zero i valori restanti";

    while(i<L-1)
    {
        yvac_inter[i]=0;
        i++;
    }

    qDebug()<<"plotto vacanze";

    setupVac(ui->vac_plot);
    ui->vac_plot->replot();


qDebug()<<"sistemo probabilità di ricombinazione";

    for(int k=0; k<L; k++)
    {
        taup_arr[k]=taup/(1+radhard_p*yvac_inter[k]*psigma*ptherm*fluence*taup);
        ptaup[k]=deltatp/taup_arr[k];
        taun_arr[k]=taun/(1+radhard_n*yvac_inter[k]*nsigma*ntherm*fluence*taun);
        ptaun[k]=deltatn/taun_arr[k];

    }


    setupTau(ui->tau_plot);
    ui->tau_plot->replot();

    text_radhard();
}

void MainWindow::on_pushButton_devsetup_clicked()
{
    FilesDialog dialog;
    dialog.exec();


    if(dialog.checkdialog==1)
    {
    path_efield=dialog.path_efield;
    path_gunn=dialog.path_gunn;

    vsatp=dialog.vsatp/100;         //in m/s
    vsatn=dialog.vsatn/100;
    temp=dialog.temp;

    mun=dialog.mun*1e-4;
    mup=dialog.mup*1e-4;
    taun=dialog.taun;
    taup=dialog.taup;

    electrostatics();


    for(int k=0; k<L; k++)
    {
        taup_arr[k]=taup/(1+radhard_p*yvac_inter[k]*psigma*ptherm*fluence*taup);
        ptaup[k]=deltatp/taup_arr[k];
        taun_arr[k]=taun/(1+radhard_n*yvac_inter[k]*nsigma*ntherm*fluence*taun);
        ptaun[k]=deltatn/taun_arr[k];

    }


    setupTau(ui->tau_plot);
    ui->tau_plot->replot();

}
    checkdialog=1;
    ui->pushButton_simsetup->setEnabled(true);
}


void MainWindow::on_pushButton_about_clicked()
{
    LicenseDialog dialog;
    dialog.exec();
}

void MainWindow::on_tau_x_valueChanged(int arg1)
{
tauplot_scale_x=arg1;
setupTau(ui->tau_plot);
ui->tau_plot->replot();

}



void MainWindow::on_tau_y_valueChanged(int arg1)
{
    tauplot_scale_y=arg1;
    setupTau(ui->tau_plot);
    ui->tau_plot->replot();

}


void MainWindow::on_checkBox_2_stateChanged(int arg1)
{
        if(ui->checkBox_2->isChecked())
        {
            ui->tau_plot->yAxis->setScaleType( QCPAxis::stLogarithmic);
            //   setupTau(ui->tau_plot);
            ui->tau_plot->replot();
        }
        else
        {
            ui->tau_plot->yAxis->setScaleType( QCPAxis::stLinear);
            //   setupTau(ui->tau_plot);
            ui->tau_plot->replot();
        }

}
